/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { inject } from '@angular/core';
/**
 * Maps an array of injectable classes with canMatch functions to an array of equivalent
 * `CanMatchFn` for use in a `Route` definition.
 *
 * Usage {@example router/utils/functional_guards.ts region='CanActivate'}
 *
 * @publicApi
 * @see {@link Route}
 */
export function mapToCanMatch(providers) {
    return providers.map(provider => (...params) => inject(provider).canMatch(...params));
}
/**
 * Maps an array of injectable classes with canActivate functions to an array of equivalent
 * `CanActivateFn` for use in a `Route` definition.
 *
 * Usage {@example router/utils/functional_guards.ts region='CanActivate'}
 *
 * @publicApi
 * @see {@link Route}
 */
export function mapToCanActivate(providers) {
    return providers.map(provider => (...params) => inject(provider).canActivate(...params));
}
/**
 * Maps an array of injectable classes with canActivateChild functions to an array of equivalent
 * `CanActivateChildFn` for use in a `Route` definition.
 *
 * Usage {@example router/utils/functional_guards.ts region='CanActivate'}
 *
 * @publicApi
 * @see {@link Route}
 */
export function mapToCanActivateChild(providers) {
    return providers.map(provider => (...params) => inject(provider).canActivateChild(...params));
}
/**
 * Maps an array of injectable classes with canDeactivate functions to an array of equivalent
 * `CanDeactivateFn` for use in a `Route` definition.
 *
 * Usage {@example router/utils/functional_guards.ts region='CanActivate'}
 *
 * @publicApi
 * @see {@link Route}
 */
export function mapToCanDeactivate(providers) {
    return providers.map(provider => (...params) => inject(provider).canDeactivate(...params));
}
/**
 * Maps an injectable class with a resolve function to an equivalent `ResolveFn`
 * for use in a `Route` definition.
 *
 * Usage {@example router/utils/functional_guards.ts region='Resolve'}
 *
 * @publicApi
 * @see {@link Route}
 */
export function mapToResolve(provider) {
    return (...params) => inject(provider).resolve(...params);
}
//# sourceMappingURL=data:application/json;base64,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