/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { runInInjectionContext } from '@angular/core';
import { EMPTY, from, of, throwError } from 'rxjs';
import { catchError, concatMap, first, map, mapTo, mergeMap, takeLast, tap } from 'rxjs/operators';
import { getInherited, hasStaticTitle } from '../router_state';
import { RouteTitleKey } from '../shared';
import { getDataKeys, wrapIntoObservable } from '../utils/collection';
import { getClosestRouteInjector } from '../utils/config';
import { getTokenOrFunctionIdentity } from '../utils/preactivation';
import { isEmptyError } from '../utils/type_guards';
export function resolveData(paramsInheritanceStrategy, injector) {
    return mergeMap(t => {
        const { targetSnapshot, guards: { canActivateChecks } } = t;
        if (!canActivateChecks.length) {
            return of(t);
        }
        // Iterating a Set in javascript  happens in insertion order so it is safe to use a `Set` to
        // preserve the correct order that the resolvers should run in.
        // https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Set#description
        const routesWithResolversToRun = new Set(canActivateChecks.map(check => check.route));
        const routesNeedingDataUpdates = new Set();
        for (const route of routesWithResolversToRun) {
            if (routesNeedingDataUpdates.has(route)) {
                continue;
            }
            // All children under the route with a resolver to run need to recompute inherited data.
            for (const newRoute of flattenRouteTree(route)) {
                routesNeedingDataUpdates.add(newRoute);
            }
        }
        let routesProcessed = 0;
        return from(routesNeedingDataUpdates)
            .pipe(concatMap(route => {
            if (routesWithResolversToRun.has(route)) {
                return runResolve(route, targetSnapshot, paramsInheritanceStrategy, injector);
            }
            else {
                route.data = getInherited(route, route.parent, paramsInheritanceStrategy).resolve;
                return of(void 0);
            }
        }), tap(() => routesProcessed++), takeLast(1), mergeMap(_ => routesProcessed === routesNeedingDataUpdates.size ? of(t) : EMPTY));
    });
}
/**
 *  Returns the `ActivatedRouteSnapshot` tree as an array, using DFS to traverse the route tree.
 */
function flattenRouteTree(route) {
    const descendants = route.children.map(child => flattenRouteTree(child)).flat();
    return [route, ...descendants];
}
function runResolve(futureARS, futureRSS, paramsInheritanceStrategy, injector) {
    const config = futureARS.routeConfig;
    const resolve = futureARS._resolve;
    if (config?.title !== undefined && !hasStaticTitle(config)) {
        resolve[RouteTitleKey] = config.title;
    }
    return resolveNode(resolve, futureARS, futureRSS, injector).pipe(map((resolvedData) => {
        futureARS._resolvedData = resolvedData;
        futureARS.data = getInherited(futureARS, futureARS.parent, paramsInheritanceStrategy).resolve;
        return null;
    }));
}
function resolveNode(resolve, futureARS, futureRSS, injector) {
    const keys = getDataKeys(resolve);
    if (keys.length === 0) {
        return of({});
    }
    const data = {};
    return from(keys).pipe(mergeMap(key => getResolver(resolve[key], futureARS, futureRSS, injector)
        .pipe(first(), tap((value) => {
        data[key] = value;
    }))), takeLast(1), mapTo(data), catchError((e) => isEmptyError(e) ? EMPTY : throwError(e)));
}
function getResolver(injectionToken, futureARS, futureRSS, injector) {
    const closestInjector = getClosestRouteInjector(futureARS) ?? injector;
    const resolver = getTokenOrFunctionIdentity(injectionToken, closestInjector);
    const resolverValue = resolver.resolve ?
        resolver.resolve(futureARS, futureRSS) :
        runInInjectionContext(closestInjector, () => resolver(futureARS, futureRSS));
    return wrapIntoObservable(resolverValue);
}
//# sourceMappingURL=data:application/json;base64,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