/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { Location } from '@angular/common';
import { EnvironmentInjector, inject, Injectable } from '@angular/core';
import { BehaviorSubject, combineLatest, EMPTY, from, of, Subject } from 'rxjs';
import { catchError, defaultIfEmpty, filter, finalize, map, switchMap, take, takeUntil, tap } from 'rxjs/operators';
import { createRouterState } from './create_router_state';
import { INPUT_BINDER } from './directives/router_outlet';
import { BeforeActivateRoutes, GuardsCheckEnd, GuardsCheckStart, IMPERATIVE_NAVIGATION, NavigationCancel, NavigationEnd, NavigationError, NavigationSkipped, NavigationStart, RedirectRequest, ResolveEnd, ResolveStart, RouteConfigLoadEnd, RouteConfigLoadStart, RoutesRecognized } from './events';
import { isNavigationCancelingError, isRedirectingNavigationCancelingError, redirectingNavigationError } from './navigation_canceling_error';
import { activateRoutes } from './operators/activate_routes';
import { checkGuards } from './operators/check_guards';
import { recognize } from './operators/recognize';
import { resolveData } from './operators/resolve_data';
import { switchTap } from './operators/switch_tap';
import { TitleStrategy } from './page_title_strategy';
import { ROUTER_CONFIGURATION } from './router_config';
import { RouterConfigLoader } from './router_config_loader';
import { ChildrenOutletContexts } from './router_outlet_context';
import { createEmptyState } from './router_state';
import { UrlHandlingStrategy } from './url_handling_strategy';
import { isUrlTree, UrlSerializer } from './url_tree';
import { getAllRouteGuards } from './utils/preactivation';
import { CREATE_VIEW_TRANSITION } from './utils/view_transition';
import * as i0 from "@angular/core";
export class NavigationTransitions {
    get hasRequestedNavigation() {
        return this.navigationId !== 0;
    }
    constructor() {
        this.currentNavigation = null;
        this.currentTransition = null;
        this.lastSuccessfulNavigation = null;
        /**
         * These events are used to communicate back to the Router about the state of the transition. The
         * Router wants to respond to these events in various ways. Because the `NavigationTransition`
         * class is not public, this event subject is not publicly exposed.
         */
        this.events = new Subject();
        /**
         * Used to abort the current transition with an error.
         */
        this.transitionAbortSubject = new Subject();
        this.configLoader = inject(RouterConfigLoader);
        this.environmentInjector = inject(EnvironmentInjector);
        this.urlSerializer = inject(UrlSerializer);
        this.rootContexts = inject(ChildrenOutletContexts);
        this.location = inject(Location);
        this.inputBindingEnabled = inject(INPUT_BINDER, { optional: true }) !== null;
        this.titleStrategy = inject(TitleStrategy);
        this.options = inject(ROUTER_CONFIGURATION, { optional: true }) || {};
        this.paramsInheritanceStrategy = this.options.paramsInheritanceStrategy || 'emptyOnly';
        this.urlHandlingStrategy = inject(UrlHandlingStrategy);
        this.createViewTransition = inject(CREATE_VIEW_TRANSITION, { optional: true });
        this.navigationId = 0;
        /**
         * Hook that enables you to pause navigation after the preactivation phase.
         * Used by `RouterModule`.
         *
         * @internal
         */
        this.afterPreactivation = () => of(void 0);
        /** @internal */
        this.rootComponentType = null;
        const onLoadStart = (r) => this.events.next(new RouteConfigLoadStart(r));
        const onLoadEnd = (r) => this.events.next(new RouteConfigLoadEnd(r));
        this.configLoader.onLoadEndListener = onLoadEnd;
        this.configLoader.onLoadStartListener = onLoadStart;
    }
    complete() {
        this.transitions?.complete();
    }
    handleNavigationRequest(request) {
        const id = ++this.navigationId;
        this.transitions?.next({ ...this.transitions.value, ...request, id });
    }
    setupNavigations(router, initialUrlTree, initialRouterState) {
        this.transitions = new BehaviorSubject({
            id: 0,
            currentUrlTree: initialUrlTree,
            currentRawUrl: initialUrlTree,
            extractedUrl: this.urlHandlingStrategy.extract(initialUrlTree),
            urlAfterRedirects: this.urlHandlingStrategy.extract(initialUrlTree),
            rawUrl: initialUrlTree,
            extras: {},
            resolve: null,
            reject: null,
            promise: Promise.resolve(true),
            source: IMPERATIVE_NAVIGATION,
            restoredState: null,
            currentSnapshot: initialRouterState.snapshot,
            targetSnapshot: null,
            currentRouterState: initialRouterState,
            targetRouterState: null,
            guards: { canActivateChecks: [], canDeactivateChecks: [] },
            guardsResult: null,
        });
        return this.transitions.pipe(filter(t => t.id !== 0), 
        // Extract URL
        map(t => ({ ...t, extractedUrl: this.urlHandlingStrategy.extract(t.rawUrl) })), 
        // Using switchMap so we cancel executing navigations when a new one comes in
        switchMap(overallTransitionState => {
            this.currentTransition = overallTransitionState;
            let completed = false;
            let errored = false;
            return of(overallTransitionState)
                .pipe(
            // Store the Navigation object
            tap(t => {
                this.currentNavigation = {
                    id: t.id,
                    initialUrl: t.rawUrl,
                    extractedUrl: t.extractedUrl,
                    trigger: t.source,
                    extras: t.extras,
                    previousNavigation: !this.lastSuccessfulNavigation ? null : {
                        ...this.lastSuccessfulNavigation,
                        previousNavigation: null,
                    },
                };
            }), switchMap(t => {
                const urlTransition = !router.navigated ||
                    this.isUpdatingInternalState() || this.isUpdatedBrowserUrl();
                const onSameUrlNavigation = t.extras.onSameUrlNavigation ?? router.onSameUrlNavigation;
                if (!urlTransition && onSameUrlNavigation !== 'reload') {
                    const reason = (typeof ngDevMode === 'undefined' || ngDevMode) ?
                        `Navigation to ${t.rawUrl} was ignored because it is the same as the current Router URL.` :
                        '';
                    this.events.next(new NavigationSkipped(t.id, this.urlSerializer.serialize(t.rawUrl), reason, 0 /* NavigationSkippedCode.IgnoredSameUrlNavigation */));
                    t.resolve(null);
                    return EMPTY;
                }
                if (this.urlHandlingStrategy.shouldProcessUrl(t.rawUrl)) {
                    return of(t).pipe(
                    // Fire NavigationStart event
                    switchMap(t => {
                        const transition = this.transitions?.getValue();
                        this.events.next(new NavigationStart(t.id, this.urlSerializer.serialize(t.extractedUrl), t.source, t.restoredState));
                        if (transition !== this.transitions?.getValue()) {
                            return EMPTY;
                        }
                        // This delay is required to match old behavior that forced
                        // navigation to always be async
                        return Promise.resolve(t);
                    }), 
                    // Recognize
                    recognize(this.environmentInjector, this.configLoader, this.rootComponentType, router.config, this.urlSerializer, this.paramsInheritanceStrategy), 
                    // Update URL if in `eager` update mode
                    tap(t => {
                        overallTransitionState.targetSnapshot = t.targetSnapshot;
                        overallTransitionState.urlAfterRedirects = t.urlAfterRedirects;
                        this.currentNavigation = {
                            ...this.currentNavigation,
                            finalUrl: t.urlAfterRedirects
                        };
                        // Fire RoutesRecognized
                        const routesRecognized = new RoutesRecognized(t.id, this.urlSerializer.serialize(t.extractedUrl), this.urlSerializer.serialize(t.urlAfterRedirects), t.targetSnapshot);
                        this.events.next(routesRecognized);
                    }));
                }
                else if (urlTransition &&
                    this.urlHandlingStrategy.shouldProcessUrl(t.currentRawUrl)) {
                    /* When the current URL shouldn't be processed, but the previous one
                     * was, we handle this "error condition" by navigating to the
                     * previously successful URL, but leaving the URL intact.*/
                    const { id, extractedUrl, source, restoredState, extras } = t;
                    const navStart = new NavigationStart(id, this.urlSerializer.serialize(extractedUrl), source, restoredState);
                    this.events.next(navStart);
                    const targetSnapshot = createEmptyState(extractedUrl, this.rootComponentType).snapshot;
                    this.currentTransition = overallTransitionState = {
                        ...t,
                        targetSnapshot,
                        urlAfterRedirects: extractedUrl,
                        extras: { ...extras, skipLocationChange: false, replaceUrl: false },
                    };
                    this.currentNavigation.finalUrl = extractedUrl;
                    return of(overallTransitionState);
                }
                else {
                    /* When neither the current or previous URL can be processed, do
                     * nothing other than update router's internal reference to the
                     * current "settled" URL. This way the next navigation will be coming
                     * from the current URL in the browser.
                     */
                    const reason = (typeof ngDevMode === 'undefined' || ngDevMode) ?
                        `Navigation was ignored because the UrlHandlingStrategy` +
                            ` indicated neither the current URL ${t.currentRawUrl} nor target URL ${t.rawUrl} should be processed.` :
                        '';
                    this.events.next(new NavigationSkipped(t.id, this.urlSerializer.serialize(t.extractedUrl), reason, 1 /* NavigationSkippedCode.IgnoredByUrlHandlingStrategy */));
                    t.resolve(null);
                    return EMPTY;
                }
            }), 
            // --- GUARDS ---
            tap(t => {
                const guardsStart = new GuardsCheckStart(t.id, this.urlSerializer.serialize(t.extractedUrl), this.urlSerializer.serialize(t.urlAfterRedirects), t.targetSnapshot);
                this.events.next(guardsStart);
            }), map(t => {
                this.currentTransition = overallTransitionState = {
                    ...t,
                    guards: getAllRouteGuards(t.targetSnapshot, t.currentSnapshot, this.rootContexts)
                };
                return overallTransitionState;
            }), checkGuards(this.environmentInjector, (evt) => this.events.next(evt)), tap(t => {
                overallTransitionState.guardsResult = t.guardsResult;
                if (isUrlTree(t.guardsResult)) {
                    throw redirectingNavigationError(this.urlSerializer, t.guardsResult);
                }
                const guardsEnd = new GuardsCheckEnd(t.id, this.urlSerializer.serialize(t.extractedUrl), this.urlSerializer.serialize(t.urlAfterRedirects), t.targetSnapshot, !!t.guardsResult);
                this.events.next(guardsEnd);
            }), filter(t => {
                if (!t.guardsResult) {
                    this.cancelNavigationTransition(t, '', 3 /* NavigationCancellationCode.GuardRejected */);
                    return false;
                }
                return true;
            }), 
            // --- RESOLVE ---
            switchTap(t => {
                if (t.guards.canActivateChecks.length) {
                    return of(t).pipe(tap(t => {
                        const resolveStart = new ResolveStart(t.id, this.urlSerializer.serialize(t.extractedUrl), this.urlSerializer.serialize(t.urlAfterRedirects), t.targetSnapshot);
                        this.events.next(resolveStart);
                    }), switchMap(t => {
                        let dataResolved = false;
                        return of(t).pipe(resolveData(this.paramsInheritanceStrategy, this.environmentInjector), tap({
                            next: () => dataResolved = true,
                            complete: () => {
                                if (!dataResolved) {
                                    this.cancelNavigationTransition(t, (typeof ngDevMode === 'undefined' || ngDevMode) ?
                                        `At least one route resolver didn't emit any value.` :
                                        '', 2 /* NavigationCancellationCode.NoDataFromResolver */);
                                }
                            }
                        }));
                    }), tap(t => {
                        const resolveEnd = new ResolveEnd(t.id, this.urlSerializer.serialize(t.extractedUrl), this.urlSerializer.serialize(t.urlAfterRedirects), t.targetSnapshot);
                        this.events.next(resolveEnd);
                    }));
                }
                return undefined;
            }), 
            // --- LOAD COMPONENTS ---
            switchTap((t) => {
                const loadComponents = (route) => {
                    const loaders = [];
                    if (route.routeConfig?.loadComponent &&
                        !route.routeConfig._loadedComponent) {
                        loaders.push(this.configLoader.loadComponent(route.routeConfig)
                            .pipe(tap(loadedComponent => {
                            route.component = loadedComponent;
                        }), map(() => void 0)));
                    }
                    for (const child of route.children) {
                        loaders.push(...loadComponents(child));
                    }
                    return loaders;
                };
                return combineLatest(loadComponents(t.targetSnapshot.root))
                    .pipe(defaultIfEmpty(), take(1));
            }), switchTap(() => this.afterPreactivation()), switchMap(() => {
                const { currentSnapshot, targetSnapshot } = overallTransitionState;
                const viewTransitionStarted = this.createViewTransition?.(this.environmentInjector, currentSnapshot.root, targetSnapshot.root);
                // If view transitions are enabled, block the navigation until the view
                // transition callback starts. Otherwise, continue immediately.
                return viewTransitionStarted ?
                    from(viewTransitionStarted).pipe(map(() => overallTransitionState)) :
                    of(overallTransitionState);
            }), map((t) => {
                const targetRouterState = createRouterState(router.routeReuseStrategy, t.targetSnapshot, t.currentRouterState);
                this.currentTransition =
                    overallTransitionState = { ...t, targetRouterState };
                this.currentNavigation.targetRouterState = targetRouterState;
                return overallTransitionState;
            }), tap(() => {
                this.events.next(new BeforeActivateRoutes());
            }), activateRoutes(this.rootContexts, router.routeReuseStrategy, (evt) => this.events.next(evt), this.inputBindingEnabled), 
            // Ensure that if some observable used to drive the transition doesn't
            // complete, the navigation still finalizes This should never happen, but
            // this is done as a safety measure to avoid surfacing this error (#49567).
            take(1), tap({
                next: (t) => {
                    completed = true;
                    this.lastSuccessfulNavigation = this.currentNavigation;
                    this.events.next(new NavigationEnd(t.id, this.urlSerializer.serialize(t.extractedUrl), this.urlSerializer.serialize(t.urlAfterRedirects)));
                    this.titleStrategy?.updateTitle(t.targetRouterState.snapshot);
                    t.resolve(true);
                },
                complete: () => {
                    completed = true;
                }
            }), 
            // There used to be a lot more logic happening directly within the
            // transition Observable. Some of this logic has been refactored out to
            // other places but there may still be errors that happen there. This gives
            // us a way to cancel the transition from the outside. This may also be
            // required in the future to support something like the abort signal of the
            // Navigation API where the navigation gets aborted from outside the
            // transition.
            takeUntil(this.transitionAbortSubject.pipe(tap(err => {
                throw err;
            }))), finalize(() => {
                /* When the navigation stream finishes either through error or success,
                 * we set the `completed` or `errored` flag. However, there are some
                 * situations where we could get here without either of those being set.
                 * For instance, a redirect during NavigationStart. Therefore, this is a
                 * catch-all to make sure the NavigationCancel event is fired when a
                 * navigation gets cancelled but not caught by other means. */
                if (!completed && !errored) {
                    const cancelationReason = (typeof ngDevMode === 'undefined' || ngDevMode) ?
                        `Navigation ID ${overallTransitionState
                            .id} is not equal to the current navigation id ${this.navigationId}` :
                        '';
                    this.cancelNavigationTransition(overallTransitionState, cancelationReason, 1 /* NavigationCancellationCode.SupersededByNewNavigation */);
                }
                // Only clear current navigation if it is still set to the one that
                // finalized.
                if (this.currentNavigation?.id === overallTransitionState.id) {
                    this.currentNavigation = null;
                }
            }), catchError((e) => {
                errored = true;
                /* This error type is issued during Redirect, and is handled as a
                 * cancellation rather than an error. */
                if (isNavigationCancelingError(e)) {
                    this.events.next(new NavigationCancel(overallTransitionState.id, this.urlSerializer.serialize(overallTransitionState.extractedUrl), e.message, e.cancellationCode));
                    // When redirecting, we need to delay resolving the navigation
                    // promise and push it to the redirect navigation
                    if (!isRedirectingNavigationCancelingError(e)) {
                        overallTransitionState.resolve(false);
                    }
                    else {
                        this.events.next(new RedirectRequest(e.url));
                    }
                    /* All other errors should reset to the router's internal URL reference
                     * to the pre-error state. */
                }
                else {
                    this.events.next(new NavigationError(overallTransitionState.id, this.urlSerializer.serialize(overallTransitionState.extractedUrl), e, overallTransitionState.targetSnapshot ?? undefined));
                    try {
                        overallTransitionState.resolve(router.errorHandler(e));
                    }
                    catch (ee) {
                        overallTransitionState.reject(ee);
                    }
                }
                return EMPTY;
            }));
            // casting because `pipe` returns observable({}) when called with 8+ arguments
        }));
    }
    cancelNavigationTransition(t, reason, code) {
        const navCancel = new NavigationCancel(t.id, this.urlSerializer.serialize(t.extractedUrl), reason, code);
        this.events.next(navCancel);
        t.resolve(false);
    }
    /**
     * @returns Whether we're navigating to somewhere that is not what the Router is
     * currently set to.
     */
    isUpdatingInternalState() {
        // TODO(atscott): The serializer should likely be used instead of
        // `UrlTree.toString()`. Custom serializers are often written to handle
        // things better than the default one (objects, for example will be
        // [Object object] with the custom serializer and be "the same" when they
        // aren't).
        // (Same for isUpdatedBrowserUrl)
        return this.currentTransition?.extractedUrl.toString() !==
            this.currentTransition?.currentUrlTree.toString();
    }
    /**
     * @returns Whether we're updating the browser URL to something new (navigation is going
     * to somewhere not displayed in the URL bar and we will update the URL
     * bar if navigation succeeds).
     */
    isUpdatedBrowserUrl() {
        // The extracted URL is the part of the URL that this application cares about. `extract` may
        // return only part of the browser URL and that part may have not changed even if some other
        // portion of the URL did.
        const extractedBrowserUrl = this.urlHandlingStrategy.extract(this.urlSerializer.parse(this.location.path(true)));
        return extractedBrowserUrl.toString() !== this.currentTransition?.extractedUrl.toString() &&
            !this.currentTransition?.extras.skipLocationChange;
    }
    static { this.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "17.0.5", ngImport: i0, type: NavigationTransitions, deps: [], target: i0.ɵɵFactoryTarget.Injectable }); }
    static { this.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "17.0.5", ngImport: i0, type: NavigationTransitions, providedIn: 'root' }); }
}
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "17.0.5", ngImport: i0, type: NavigationTransitions, decorators: [{
            type: Injectable,
            args: [{ providedIn: 'root' }]
        }], ctorParameters: () => [] });
export function isBrowserTriggeredNavigation(source) {
    return source !== IMPERATIVE_NAVIGATION;
}
//# sourceMappingURL=data:application/json;base64,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