/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { FocusMonitor, FocusTrapFactory, InteractivityChecker } from '@angular/cdk/a11y';
import { OverlayRef } from '@angular/cdk/overlay';
import { DOCUMENT } from '@angular/common';
import { ChangeDetectionStrategy, Component, ElementRef, EventEmitter, Inject, NgZone, Optional, ViewEncapsulation, } from '@angular/core';
import { MatDialogConfig } from './dialog-config';
import { ANIMATION_MODULE_TYPE } from '@angular/platform-browser/animations';
import { CdkDialogContainer } from '@angular/cdk/dialog';
import { coerceNumberProperty } from '@angular/cdk/coercion';
import { PortalModule } from '@angular/cdk/portal';
import * as i0 from "@angular/core";
import * as i1 from "@angular/cdk/a11y";
import * as i2 from "./dialog-config";
import * as i3 from "@angular/cdk/overlay";
import * as i4 from "@angular/cdk/portal";
/** Class added when the dialog is open. */
const OPEN_CLASS = 'mdc-dialog--open';
/** Class added while the dialog is opening. */
const OPENING_CLASS = 'mdc-dialog--opening';
/** Class added while the dialog is closing. */
const CLOSING_CLASS = 'mdc-dialog--closing';
/** Duration of the opening animation in milliseconds. */
export const OPEN_ANIMATION_DURATION = 150;
/** Duration of the closing animation in milliseconds. */
export const CLOSE_ANIMATION_DURATION = 75;
export class MatDialogContainer extends CdkDialogContainer {
    constructor(elementRef, focusTrapFactory, _document, dialogConfig, interactivityChecker, ngZone, overlayRef, _animationMode, focusMonitor) {
        super(elementRef, focusTrapFactory, _document, dialogConfig, interactivityChecker, ngZone, overlayRef, focusMonitor);
        this._animationMode = _animationMode;
        /** Emits when an animation state changes. */
        this._animationStateChanged = new EventEmitter();
        /** Whether animations are enabled. */
        this._animationsEnabled = this._animationMode !== 'NoopAnimations';
        /** Host element of the dialog container component. */
        this._hostElement = this._elementRef.nativeElement;
        /** Duration of the dialog open animation. */
        this._enterAnimationDuration = this._animationsEnabled
            ? parseCssTime(this._config.enterAnimationDuration) ?? OPEN_ANIMATION_DURATION
            : 0;
        /** Duration of the dialog close animation. */
        this._exitAnimationDuration = this._animationsEnabled
            ? parseCssTime(this._config.exitAnimationDuration) ?? CLOSE_ANIMATION_DURATION
            : 0;
        /** Current timer for dialog animations. */
        this._animationTimer = null;
        /**
         * Completes the dialog open by clearing potential animation classes, trapping
         * focus and emitting an opened event.
         */
        this._finishDialogOpen = () => {
            this._clearAnimationClasses();
            this._openAnimationDone(this._enterAnimationDuration);
        };
        /**
         * Completes the dialog close by clearing potential animation classes, restoring
         * focus and emitting a closed event.
         */
        this._finishDialogClose = () => {
            this._clearAnimationClasses();
            this._animationStateChanged.emit({ state: 'closed', totalTime: this._exitAnimationDuration });
        };
    }
    _contentAttached() {
        // Delegate to the original dialog-container initialization (i.e. saving the
        // previous element, setting up the focus trap and moving focus to the container).
        super._contentAttached();
        // Note: Usually we would be able to use the MDC dialog foundation here to handle
        // the dialog animation for us, but there are a few reasons why we just leverage
        // their styles and not use the runtime foundation code:
        //   1. Foundation does not allow us to disable animations.
        //   2. Foundation contains unnecessary features we don't need and aren't
        //      tree-shakeable. e.g. background scrim, keyboard event handlers for ESC button.
        //   3. Foundation uses unnecessary timers for animations to work around limitations
        //      in React's `setState` mechanism.
        //      https://github.com/material-components/material-components-web/pull/3682.
        this._startOpenAnimation();
    }
    /** Starts the dialog open animation if enabled. */
    _startOpenAnimation() {
        this._animationStateChanged.emit({ state: 'opening', totalTime: this._enterAnimationDuration });
        if (this._animationsEnabled) {
            this._hostElement.style.setProperty(TRANSITION_DURATION_PROPERTY, `${this._enterAnimationDuration}ms`);
            // We need to give the `setProperty` call from above some time to be applied.
            // One would expect that the open class is added once the animation finished, but MDC
            // uses the open class in combination with the opening class to start the animation.
            this._requestAnimationFrame(() => this._hostElement.classList.add(OPENING_CLASS, OPEN_CLASS));
            this._waitForAnimationToComplete(this._enterAnimationDuration, this._finishDialogOpen);
        }
        else {
            this._hostElement.classList.add(OPEN_CLASS);
            // Note: We could immediately finish the dialog opening here with noop animations,
            // but we defer until next tick so that consumers can subscribe to `afterOpened`.
            // Executing this immediately would mean that `afterOpened` emits synchronously
            // on `dialog.open` before the consumer had a change to subscribe to `afterOpened`.
            Promise.resolve().then(() => this._finishDialogOpen());
        }
    }
    /**
     * Starts the exit animation of the dialog if enabled. This method is
     * called by the dialog ref.
     */
    _startExitAnimation() {
        this._animationStateChanged.emit({ state: 'closing', totalTime: this._exitAnimationDuration });
        this._hostElement.classList.remove(OPEN_CLASS);
        if (this._animationsEnabled) {
            this._hostElement.style.setProperty(TRANSITION_DURATION_PROPERTY, `${this._exitAnimationDuration}ms`);
            // We need to give the `setProperty` call from above some time to be applied.
            this._requestAnimationFrame(() => this._hostElement.classList.add(CLOSING_CLASS));
            this._waitForAnimationToComplete(this._exitAnimationDuration, this._finishDialogClose);
        }
        else {
            // This subscription to the `OverlayRef#backdropClick` observable in the `DialogRef` is
            // set up before any user can subscribe to the backdrop click. The subscription triggers
            // the dialog close and this method synchronously. If we'd synchronously emit the `CLOSED`
            // animation state event if animations are disabled, the overlay would be disposed
            // immediately and all other subscriptions to `DialogRef#backdropClick` would be silently
            // skipped. We work around this by waiting with the dialog close until the next tick when
            // all subscriptions have been fired as expected. This is not an ideal solution, but
            // there doesn't seem to be any other good way. Alternatives that have been considered:
            //   1. Deferring `DialogRef.close`. This could be a breaking change due to a new microtask.
            //      Also this issue is specific to the MDC implementation where the dialog could
            //      technically be closed synchronously. In the non-MDC one, Angular animations are used
            //      and closing always takes at least a tick.
            //   2. Ensuring that user subscriptions to `backdropClick`, `keydownEvents` in the dialog
            //      ref are first. This would solve the issue, but has the risk of memory leaks and also
            //      doesn't solve the case where consumers call `DialogRef.close` in their subscriptions.
            // Based on the fact that this is specific to the MDC-based implementation of the dialog
            // animations, the defer is applied here.
            Promise.resolve().then(() => this._finishDialogClose());
        }
    }
    /** Clears all dialog animation classes. */
    _clearAnimationClasses() {
        this._hostElement.classList.remove(OPENING_CLASS, CLOSING_CLASS);
    }
    _waitForAnimationToComplete(duration, callback) {
        if (this._animationTimer !== null) {
            clearTimeout(this._animationTimer);
        }
        // Note that we want this timer to run inside the NgZone, because we want
        // the related events like `afterClosed` to be inside the zone as well.
        this._animationTimer = setTimeout(callback, duration);
    }
    /** Runs a callback in `requestAnimationFrame`, if available. */
    _requestAnimationFrame(callback) {
        this._ngZone.runOutsideAngular(() => {
            if (typeof requestAnimationFrame === 'function') {
                requestAnimationFrame(callback);
            }
            else {
                callback();
            }
        });
    }
    _captureInitialFocus() {
        if (!this._config.delayFocusTrap) {
            this._trapFocus();
        }
    }
    /**
     * Callback for when the open dialog animation has finished. Intended to
     * be called by sub-classes that use different animation implementations.
     */
    _openAnimationDone(totalTime) {
        if (this._config.delayFocusTrap) {
            this._trapFocus();
        }
        this._animationStateChanged.next({ state: 'opened', totalTime });
    }
    ngOnDestroy() {
        super.ngOnDestroy();
        if (this._animationTimer !== null) {
            clearTimeout(this._animationTimer);
        }
    }
    attachComponentPortal(portal) {
        // When a component is passed into the dialog, the host element interrupts
        // the `display:flex` from affecting the dialog title, content, and
        // actions. To fix this, we make the component host `display: contents` by
        // marking its host with the `mat-mdc-dialog-component-host` class.
        //
        // Note that this problem does not exist when a template ref is used since
        // the title, contents, and actions are then nested directly under the
        // dialog surface.
        const ref = super.attachComponentPortal(portal);
        ref.location.nativeElement.classList.add('mat-mdc-dialog-component-host');
        return ref;
    }
    static { this.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "17.0.0", ngImport: i0, type: MatDialogContainer, deps: [{ token: i0.ElementRef }, { token: i1.FocusTrapFactory }, { token: DOCUMENT, optional: true }, { token: i2.MatDialogConfig }, { token: i1.InteractivityChecker }, { token: i0.NgZone }, { token: i3.OverlayRef }, { token: ANIMATION_MODULE_TYPE, optional: true }, { token: i1.FocusMonitor }], target: i0.ɵɵFactoryTarget.Component }); }
    static { this.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "14.0.0", version: "17.0.0", type: MatDialogContainer, isStandalone: true, selector: "mat-dialog-container", host: { attributes: { "tabindex": "-1" }, properties: { "attr.aria-modal": "_config.ariaModal", "id": "_config.id", "attr.role": "_config.role", "attr.aria-labelledby": "_config.ariaLabel ? null : _ariaLabelledByQueue[0]", "attr.aria-label": "_config.ariaLabel", "attr.aria-describedby": "_config.ariaDescribedBy || null", "class._mat-animation-noopable": "!_animationsEnabled" }, classAttribute: "mat-mdc-dialog-container mdc-dialog" }, usesInheritance: true, ngImport: i0, template: "<div class=\"mdc-dialog__container\">\n  <div class=\"mat-mdc-dialog-surface mdc-dialog__surface\">\n    <ng-template cdkPortalOutlet></ng-template>\n  </div>\n</div>\n", styles: [".mdc-elevation-overlay{position:absolute;border-radius:inherit;pointer-events:none;opacity:var(--mdc-elevation-overlay-opacity, 0);transition:opacity 280ms cubic-bezier(0.4, 0, 0.2, 1)}.mdc-dialog,.mdc-dialog__scrim{position:fixed;top:0;left:0;align-items:center;justify-content:center;box-sizing:border-box;width:100%;height:100%}.mdc-dialog{display:none;z-index:var(--mdc-dialog-z-index, 7)}.mdc-dialog .mdc-dialog__content{padding:20px 24px 20px 24px}.mdc-dialog .mdc-dialog__surface{min-width:280px}@media(max-width: 592px){.mdc-dialog .mdc-dialog__surface{max-width:calc(100vw - 32px)}}@media(min-width: 592px){.mdc-dialog .mdc-dialog__surface{max-width:560px}}.mdc-dialog .mdc-dialog__surface{max-height:calc(100% - 32px)}.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface{max-width:none}@media(max-width: 960px){.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface{max-height:560px;width:560px}.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface .mdc-dialog__close{right:-12px}}@media(max-width: 720px)and (max-width: 672px){.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface{width:calc(100vw - 112px)}}@media(max-width: 720px)and (min-width: 672px){.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface{width:560px}}@media(max-width: 720px)and (max-height: 720px){.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface{max-height:calc(100vh - 160px)}}@media(max-width: 720px)and (min-height: 720px){.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface{max-height:560px}}@media(max-width: 720px){.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface .mdc-dialog__close{right:-12px}}@media(max-width: 720px)and (max-height: 400px),(max-width: 600px),(min-width: 720px)and (max-height: 400px){.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface{height:100%;max-height:100vh;max-width:100vw;width:100vw;border-radius:0}.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface .mdc-dialog__close{order:-1;left:-12px}.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface .mdc-dialog__header{padding:0 16px 9px;justify-content:flex-start}.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface .mdc-dialog__title{margin-left:calc(16px - 2 * 12px)}}@media(min-width: 960px){.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface{width:calc(100vw - 400px)}.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface .mdc-dialog__close{right:-12px}}.mdc-dialog.mdc-dialog__scrim--hidden .mdc-dialog__scrim{opacity:0}.mdc-dialog__scrim{opacity:0;z-index:-1}.mdc-dialog__container{display:flex;flex-direction:row;align-items:center;justify-content:space-around;box-sizing:border-box;height:100%;transform:scale(0.8);opacity:0;pointer-events:none}.mdc-dialog__surface{position:relative;display:flex;flex-direction:column;flex-grow:0;flex-shrink:0;box-sizing:border-box;max-width:100%;max-height:100%;pointer-events:auto;overflow-y:auto;outline:0}.mdc-dialog__surface .mdc-elevation-overlay{width:100%;height:100%;top:0;left:0}[dir=rtl] .mdc-dialog__surface,.mdc-dialog__surface[dir=rtl]{text-align:right}@media screen and (forced-colors: active),(-ms-high-contrast: active){.mdc-dialog__surface{outline:2px solid windowText}}.mdc-dialog__surface::before{position:absolute;box-sizing:border-box;width:100%;height:100%;top:0;left:0;border:2px solid rgba(0,0,0,0);border-radius:inherit;content:\"\";pointer-events:none}@media screen and (forced-colors: active){.mdc-dialog__surface::before{border-color:CanvasText}}@media screen and (-ms-high-contrast: active),screen and (-ms-high-contrast: none){.mdc-dialog__surface::before{content:none}}.mdc-dialog__title{display:block;margin-top:0;position:relative;flex-shrink:0;box-sizing:border-box;margin:0 0 1px;padding:0 24px 9px}.mdc-dialog__title::before{display:inline-block;width:0;height:40px;content:\"\";vertical-align:0}[dir=rtl] .mdc-dialog__title,.mdc-dialog__title[dir=rtl]{text-align:right}.mdc-dialog--scrollable .mdc-dialog__title{margin-bottom:1px;padding-bottom:15px}.mdc-dialog--fullscreen .mdc-dialog__header{align-items:baseline;border-bottom:1px solid rgba(0,0,0,0);display:inline-flex;justify-content:space-between;padding:0 24px 9px;z-index:1}@media screen and (forced-colors: active){.mdc-dialog--fullscreen .mdc-dialog__header{border-bottom-color:CanvasText}}.mdc-dialog--fullscreen .mdc-dialog__header .mdc-dialog__close{right:-12px}.mdc-dialog--fullscreen .mdc-dialog__title{margin-bottom:0;padding:0;border-bottom:0}.mdc-dialog--fullscreen.mdc-dialog--scrollable .mdc-dialog__title{border-bottom:0;margin-bottom:0}.mdc-dialog--fullscreen .mdc-dialog__close{top:5px}.mdc-dialog--fullscreen.mdc-dialog--scrollable .mdc-dialog__actions{border-top:1px solid rgba(0,0,0,0)}@media screen and (forced-colors: active){.mdc-dialog--fullscreen.mdc-dialog--scrollable .mdc-dialog__actions{border-top-color:CanvasText}}.mdc-dialog--fullscreen--titleless .mdc-dialog__close{margin-top:4px}.mdc-dialog--fullscreen--titleless.mdc-dialog--scrollable .mdc-dialog__close{margin-top:0}.mdc-dialog__content{flex-grow:1;box-sizing:border-box;margin:0;overflow:auto}.mdc-dialog__content>:first-child{margin-top:0}.mdc-dialog__content>:last-child{margin-bottom:0}.mdc-dialog__title+.mdc-dialog__content,.mdc-dialog__header+.mdc-dialog__content{padding-top:0}.mdc-dialog--scrollable .mdc-dialog__title+.mdc-dialog__content{padding-top:8px;padding-bottom:8px}.mdc-dialog__content .mdc-deprecated-list:first-child:last-child{padding:6px 0 0}.mdc-dialog--scrollable .mdc-dialog__content .mdc-deprecated-list:first-child:last-child{padding:0}.mdc-dialog__actions{display:flex;position:relative;flex-shrink:0;flex-wrap:wrap;align-items:center;justify-content:flex-end;box-sizing:border-box;min-height:52px;margin:0;padding:8px;border-top:1px solid rgba(0,0,0,0)}@media screen and (forced-colors: active){.mdc-dialog__actions{border-top-color:CanvasText}}.mdc-dialog--stacked .mdc-dialog__actions{flex-direction:column;align-items:flex-end}.mdc-dialog__button{margin-left:8px;margin-right:0;max-width:100%;text-align:right}[dir=rtl] .mdc-dialog__button,.mdc-dialog__button[dir=rtl]{margin-left:0;margin-right:8px}.mdc-dialog__button:first-child{margin-left:0;margin-right:0}[dir=rtl] .mdc-dialog__button:first-child,.mdc-dialog__button:first-child[dir=rtl]{margin-left:0;margin-right:0}[dir=rtl] .mdc-dialog__button,.mdc-dialog__button[dir=rtl]{text-align:left}.mdc-dialog--stacked .mdc-dialog__button:not(:first-child){margin-top:12px}.mdc-dialog--open,.mdc-dialog--opening,.mdc-dialog--closing{display:flex}.mdc-dialog--opening .mdc-dialog__scrim{transition:opacity 150ms linear}.mdc-dialog--opening .mdc-dialog__container{transition:opacity 75ms linear,transform 150ms 0ms cubic-bezier(0, 0, 0.2, 1)}.mdc-dialog--closing .mdc-dialog__scrim,.mdc-dialog--closing .mdc-dialog__container{transition:opacity 75ms linear}.mdc-dialog--closing .mdc-dialog__container{transform:none}.mdc-dialog--open .mdc-dialog__scrim{opacity:1}.mdc-dialog--open .mdc-dialog__container{transform:none;opacity:1}.mdc-dialog--open.mdc-dialog__surface-scrim--shown .mdc-dialog__surface-scrim{opacity:1}.mdc-dialog--open.mdc-dialog__surface-scrim--hiding .mdc-dialog__surface-scrim{transition:opacity 75ms linear}.mdc-dialog--open.mdc-dialog__surface-scrim--showing .mdc-dialog__surface-scrim{transition:opacity 150ms linear}.mdc-dialog__surface-scrim{display:none;opacity:0;position:absolute;width:100%;height:100%;z-index:1}.mdc-dialog__surface-scrim--shown .mdc-dialog__surface-scrim,.mdc-dialog__surface-scrim--showing .mdc-dialog__surface-scrim,.mdc-dialog__surface-scrim--hiding .mdc-dialog__surface-scrim{display:block}.mdc-dialog-scroll-lock{overflow:hidden}.mdc-dialog--no-content-padding .mdc-dialog__content{padding:0}.mdc-dialog--sheet .mdc-dialog__container .mdc-dialog__close{right:12px;top:9px;position:absolute;z-index:1}.mdc-dialog__scrim--removed{pointer-events:none}.mdc-dialog__scrim--removed .mdc-dialog__scrim,.mdc-dialog__scrim--removed .mdc-dialog__surface-scrim{display:none}.mat-mdc-dialog-content{max-height:65vh}.mat-mdc-dialog-container{position:static;display:block}.mat-mdc-dialog-container,.mat-mdc-dialog-container .mdc-dialog__container,.mat-mdc-dialog-container .mdc-dialog__surface{max-height:inherit;min-height:inherit;min-width:inherit;max-width:inherit}.mat-mdc-dialog-container .mdc-dialog__surface{width:100%;height:100%}.mat-mdc-dialog-component-host{display:contents}.mat-mdc-dialog-container{--mdc-dialog-container-elevation: var(--mdc-dialog-container-elevation-shadow);outline:0}.mat-mdc-dialog-container .mdc-dialog__surface{background-color:var(--mdc-dialog-container-color, white)}.mat-mdc-dialog-container .mdc-dialog__surface{box-shadow:var(--mdc-dialog-container-elevation, 0px 11px 15px -7px rgba(0, 0, 0, 0.2), 0px 24px 38px 3px rgba(0, 0, 0, 0.14), 0px 9px 46px 8px rgba(0, 0, 0, 0.12))}.mat-mdc-dialog-container .mdc-dialog__surface{border-radius:var(--mdc-dialog-container-shape, 4px)}.mat-mdc-dialog-container .mdc-dialog__title{font-family:var(--mdc-dialog-subhead-font, Roboto, sans-serif);line-height:var(--mdc-dialog-subhead-line-height, 1.5rem);font-size:var(--mdc-dialog-subhead-size, 1rem);font-weight:var(--mdc-dialog-subhead-weight, 400);letter-spacing:var(--mdc-dialog-subhead-tracking, 0.03125em)}.mat-mdc-dialog-container .mdc-dialog__title{color:var(--mdc-dialog-subhead-color, rgba(0, 0, 0, 0.87))}.mat-mdc-dialog-container .mdc-dialog__content{font-family:var(--mdc-dialog-supporting-text-font, Roboto, sans-serif);line-height:var(--mdc-dialog-supporting-text-line-height, 1.5rem);font-size:var(--mdc-dialog-supporting-text-size, 1rem);font-weight:var(--mdc-dialog-supporting-text-weight, 400);letter-spacing:var(--mdc-dialog-supporting-text-tracking, 0.03125em)}.mat-mdc-dialog-container .mdc-dialog__content{color:var(--mdc-dialog-supporting-text-color, rgba(0, 0, 0, 0.6))}.mat-mdc-dialog-container .mdc-dialog__container{transition-duration:var(--mat-dialog-transition-duration, 0ms)}.mat-mdc-dialog-container._mat-animation-noopable .mdc-dialog__container{transition:none}.mat-mdc-dialog-content{display:block}.mat-mdc-dialog-actions{justify-content:start}.mat-mdc-dialog-actions.mat-mdc-dialog-actions-align-center,.mat-mdc-dialog-actions[align=center]{justify-content:center}.mat-mdc-dialog-actions.mat-mdc-dialog-actions-align-end,.mat-mdc-dialog-actions[align=end]{justify-content:flex-end}.mat-mdc-dialog-actions .mat-button-base+.mat-button-base,.mat-mdc-dialog-actions .mat-mdc-button-base+.mat-mdc-button-base{margin-left:8px}[dir=rtl] .mat-mdc-dialog-actions .mat-button-base+.mat-button-base,[dir=rtl] .mat-mdc-dialog-actions .mat-mdc-button-base+.mat-mdc-button-base{margin-left:0;margin-right:8px}"], dependencies: [{ kind: "ngmodule", type: PortalModule }, { kind: "directive", type: i4.CdkPortalOutlet, selector: "[cdkPortalOutlet]", inputs: ["cdkPortalOutlet"], outputs: ["attached"], exportAs: ["cdkPortalOutlet"] }], changeDetection: i0.ChangeDetectionStrategy.Default, encapsulation: i0.ViewEncapsulation.None }); }
}
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "17.0.0", ngImport: i0, type: MatDialogContainer, decorators: [{
            type: Component,
            args: [{ selector: 'mat-dialog-container', encapsulation: ViewEncapsulation.None, changeDetection: ChangeDetectionStrategy.Default, standalone: true, imports: [PortalModule], host: {
                        'class': 'mat-mdc-dialog-container mdc-dialog',
                        'tabindex': '-1',
                        '[attr.aria-modal]': '_config.ariaModal',
                        '[id]': '_config.id',
                        '[attr.role]': '_config.role',
                        '[attr.aria-labelledby]': '_config.ariaLabel ? null : _ariaLabelledByQueue[0]',
                        '[attr.aria-label]': '_config.ariaLabel',
                        '[attr.aria-describedby]': '_config.ariaDescribedBy || null',
                        '[class._mat-animation-noopable]': '!_animationsEnabled',
                    }, template: "<div class=\"mdc-dialog__container\">\n  <div class=\"mat-mdc-dialog-surface mdc-dialog__surface\">\n    <ng-template cdkPortalOutlet></ng-template>\n  </div>\n</div>\n", styles: [".mdc-elevation-overlay{position:absolute;border-radius:inherit;pointer-events:none;opacity:var(--mdc-elevation-overlay-opacity, 0);transition:opacity 280ms cubic-bezier(0.4, 0, 0.2, 1)}.mdc-dialog,.mdc-dialog__scrim{position:fixed;top:0;left:0;align-items:center;justify-content:center;box-sizing:border-box;width:100%;height:100%}.mdc-dialog{display:none;z-index:var(--mdc-dialog-z-index, 7)}.mdc-dialog .mdc-dialog__content{padding:20px 24px 20px 24px}.mdc-dialog .mdc-dialog__surface{min-width:280px}@media(max-width: 592px){.mdc-dialog .mdc-dialog__surface{max-width:calc(100vw - 32px)}}@media(min-width: 592px){.mdc-dialog .mdc-dialog__surface{max-width:560px}}.mdc-dialog .mdc-dialog__surface{max-height:calc(100% - 32px)}.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface{max-width:none}@media(max-width: 960px){.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface{max-height:560px;width:560px}.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface .mdc-dialog__close{right:-12px}}@media(max-width: 720px)and (max-width: 672px){.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface{width:calc(100vw - 112px)}}@media(max-width: 720px)and (min-width: 672px){.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface{width:560px}}@media(max-width: 720px)and (max-height: 720px){.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface{max-height:calc(100vh - 160px)}}@media(max-width: 720px)and (min-height: 720px){.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface{max-height:560px}}@media(max-width: 720px){.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface .mdc-dialog__close{right:-12px}}@media(max-width: 720px)and (max-height: 400px),(max-width: 600px),(min-width: 720px)and (max-height: 400px){.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface{height:100%;max-height:100vh;max-width:100vw;width:100vw;border-radius:0}.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface .mdc-dialog__close{order:-1;left:-12px}.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface .mdc-dialog__header{padding:0 16px 9px;justify-content:flex-start}.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface .mdc-dialog__title{margin-left:calc(16px - 2 * 12px)}}@media(min-width: 960px){.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface{width:calc(100vw - 400px)}.mdc-dialog.mdc-dialog--fullscreen .mdc-dialog__surface .mdc-dialog__close{right:-12px}}.mdc-dialog.mdc-dialog__scrim--hidden .mdc-dialog__scrim{opacity:0}.mdc-dialog__scrim{opacity:0;z-index:-1}.mdc-dialog__container{display:flex;flex-direction:row;align-items:center;justify-content:space-around;box-sizing:border-box;height:100%;transform:scale(0.8);opacity:0;pointer-events:none}.mdc-dialog__surface{position:relative;display:flex;flex-direction:column;flex-grow:0;flex-shrink:0;box-sizing:border-box;max-width:100%;max-height:100%;pointer-events:auto;overflow-y:auto;outline:0}.mdc-dialog__surface .mdc-elevation-overlay{width:100%;height:100%;top:0;left:0}[dir=rtl] .mdc-dialog__surface,.mdc-dialog__surface[dir=rtl]{text-align:right}@media screen and (forced-colors: active),(-ms-high-contrast: active){.mdc-dialog__surface{outline:2px solid windowText}}.mdc-dialog__surface::before{position:absolute;box-sizing:border-box;width:100%;height:100%;top:0;left:0;border:2px solid rgba(0,0,0,0);border-radius:inherit;content:\"\";pointer-events:none}@media screen and (forced-colors: active){.mdc-dialog__surface::before{border-color:CanvasText}}@media screen and (-ms-high-contrast: active),screen and (-ms-high-contrast: none){.mdc-dialog__surface::before{content:none}}.mdc-dialog__title{display:block;margin-top:0;position:relative;flex-shrink:0;box-sizing:border-box;margin:0 0 1px;padding:0 24px 9px}.mdc-dialog__title::before{display:inline-block;width:0;height:40px;content:\"\";vertical-align:0}[dir=rtl] .mdc-dialog__title,.mdc-dialog__title[dir=rtl]{text-align:right}.mdc-dialog--scrollable .mdc-dialog__title{margin-bottom:1px;padding-bottom:15px}.mdc-dialog--fullscreen .mdc-dialog__header{align-items:baseline;border-bottom:1px solid rgba(0,0,0,0);display:inline-flex;justify-content:space-between;padding:0 24px 9px;z-index:1}@media screen and (forced-colors: active){.mdc-dialog--fullscreen .mdc-dialog__header{border-bottom-color:CanvasText}}.mdc-dialog--fullscreen .mdc-dialog__header .mdc-dialog__close{right:-12px}.mdc-dialog--fullscreen .mdc-dialog__title{margin-bottom:0;padding:0;border-bottom:0}.mdc-dialog--fullscreen.mdc-dialog--scrollable .mdc-dialog__title{border-bottom:0;margin-bottom:0}.mdc-dialog--fullscreen .mdc-dialog__close{top:5px}.mdc-dialog--fullscreen.mdc-dialog--scrollable .mdc-dialog__actions{border-top:1px solid rgba(0,0,0,0)}@media screen and (forced-colors: active){.mdc-dialog--fullscreen.mdc-dialog--scrollable .mdc-dialog__actions{border-top-color:CanvasText}}.mdc-dialog--fullscreen--titleless .mdc-dialog__close{margin-top:4px}.mdc-dialog--fullscreen--titleless.mdc-dialog--scrollable .mdc-dialog__close{margin-top:0}.mdc-dialog__content{flex-grow:1;box-sizing:border-box;margin:0;overflow:auto}.mdc-dialog__content>:first-child{margin-top:0}.mdc-dialog__content>:last-child{margin-bottom:0}.mdc-dialog__title+.mdc-dialog__content,.mdc-dialog__header+.mdc-dialog__content{padding-top:0}.mdc-dialog--scrollable .mdc-dialog__title+.mdc-dialog__content{padding-top:8px;padding-bottom:8px}.mdc-dialog__content .mdc-deprecated-list:first-child:last-child{padding:6px 0 0}.mdc-dialog--scrollable .mdc-dialog__content .mdc-deprecated-list:first-child:last-child{padding:0}.mdc-dialog__actions{display:flex;position:relative;flex-shrink:0;flex-wrap:wrap;align-items:center;justify-content:flex-end;box-sizing:border-box;min-height:52px;margin:0;padding:8px;border-top:1px solid rgba(0,0,0,0)}@media screen and (forced-colors: active){.mdc-dialog__actions{border-top-color:CanvasText}}.mdc-dialog--stacked .mdc-dialog__actions{flex-direction:column;align-items:flex-end}.mdc-dialog__button{margin-left:8px;margin-right:0;max-width:100%;text-align:right}[dir=rtl] .mdc-dialog__button,.mdc-dialog__button[dir=rtl]{margin-left:0;margin-right:8px}.mdc-dialog__button:first-child{margin-left:0;margin-right:0}[dir=rtl] .mdc-dialog__button:first-child,.mdc-dialog__button:first-child[dir=rtl]{margin-left:0;margin-right:0}[dir=rtl] .mdc-dialog__button,.mdc-dialog__button[dir=rtl]{text-align:left}.mdc-dialog--stacked .mdc-dialog__button:not(:first-child){margin-top:12px}.mdc-dialog--open,.mdc-dialog--opening,.mdc-dialog--closing{display:flex}.mdc-dialog--opening .mdc-dialog__scrim{transition:opacity 150ms linear}.mdc-dialog--opening .mdc-dialog__container{transition:opacity 75ms linear,transform 150ms 0ms cubic-bezier(0, 0, 0.2, 1)}.mdc-dialog--closing .mdc-dialog__scrim,.mdc-dialog--closing .mdc-dialog__container{transition:opacity 75ms linear}.mdc-dialog--closing .mdc-dialog__container{transform:none}.mdc-dialog--open .mdc-dialog__scrim{opacity:1}.mdc-dialog--open .mdc-dialog__container{transform:none;opacity:1}.mdc-dialog--open.mdc-dialog__surface-scrim--shown .mdc-dialog__surface-scrim{opacity:1}.mdc-dialog--open.mdc-dialog__surface-scrim--hiding .mdc-dialog__surface-scrim{transition:opacity 75ms linear}.mdc-dialog--open.mdc-dialog__surface-scrim--showing .mdc-dialog__surface-scrim{transition:opacity 150ms linear}.mdc-dialog__surface-scrim{display:none;opacity:0;position:absolute;width:100%;height:100%;z-index:1}.mdc-dialog__surface-scrim--shown .mdc-dialog__surface-scrim,.mdc-dialog__surface-scrim--showing .mdc-dialog__surface-scrim,.mdc-dialog__surface-scrim--hiding .mdc-dialog__surface-scrim{display:block}.mdc-dialog-scroll-lock{overflow:hidden}.mdc-dialog--no-content-padding .mdc-dialog__content{padding:0}.mdc-dialog--sheet .mdc-dialog__container .mdc-dialog__close{right:12px;top:9px;position:absolute;z-index:1}.mdc-dialog__scrim--removed{pointer-events:none}.mdc-dialog__scrim--removed .mdc-dialog__scrim,.mdc-dialog__scrim--removed .mdc-dialog__surface-scrim{display:none}.mat-mdc-dialog-content{max-height:65vh}.mat-mdc-dialog-container{position:static;display:block}.mat-mdc-dialog-container,.mat-mdc-dialog-container .mdc-dialog__container,.mat-mdc-dialog-container .mdc-dialog__surface{max-height:inherit;min-height:inherit;min-width:inherit;max-width:inherit}.mat-mdc-dialog-container .mdc-dialog__surface{width:100%;height:100%}.mat-mdc-dialog-component-host{display:contents}.mat-mdc-dialog-container{--mdc-dialog-container-elevation: var(--mdc-dialog-container-elevation-shadow);outline:0}.mat-mdc-dialog-container .mdc-dialog__surface{background-color:var(--mdc-dialog-container-color, white)}.mat-mdc-dialog-container .mdc-dialog__surface{box-shadow:var(--mdc-dialog-container-elevation, 0px 11px 15px -7px rgba(0, 0, 0, 0.2), 0px 24px 38px 3px rgba(0, 0, 0, 0.14), 0px 9px 46px 8px rgba(0, 0, 0, 0.12))}.mat-mdc-dialog-container .mdc-dialog__surface{border-radius:var(--mdc-dialog-container-shape, 4px)}.mat-mdc-dialog-container .mdc-dialog__title{font-family:var(--mdc-dialog-subhead-font, Roboto, sans-serif);line-height:var(--mdc-dialog-subhead-line-height, 1.5rem);font-size:var(--mdc-dialog-subhead-size, 1rem);font-weight:var(--mdc-dialog-subhead-weight, 400);letter-spacing:var(--mdc-dialog-subhead-tracking, 0.03125em)}.mat-mdc-dialog-container .mdc-dialog__title{color:var(--mdc-dialog-subhead-color, rgba(0, 0, 0, 0.87))}.mat-mdc-dialog-container .mdc-dialog__content{font-family:var(--mdc-dialog-supporting-text-font, Roboto, sans-serif);line-height:var(--mdc-dialog-supporting-text-line-height, 1.5rem);font-size:var(--mdc-dialog-supporting-text-size, 1rem);font-weight:var(--mdc-dialog-supporting-text-weight, 400);letter-spacing:var(--mdc-dialog-supporting-text-tracking, 0.03125em)}.mat-mdc-dialog-container .mdc-dialog__content{color:var(--mdc-dialog-supporting-text-color, rgba(0, 0, 0, 0.6))}.mat-mdc-dialog-container .mdc-dialog__container{transition-duration:var(--mat-dialog-transition-duration, 0ms)}.mat-mdc-dialog-container._mat-animation-noopable .mdc-dialog__container{transition:none}.mat-mdc-dialog-content{display:block}.mat-mdc-dialog-actions{justify-content:start}.mat-mdc-dialog-actions.mat-mdc-dialog-actions-align-center,.mat-mdc-dialog-actions[align=center]{justify-content:center}.mat-mdc-dialog-actions.mat-mdc-dialog-actions-align-end,.mat-mdc-dialog-actions[align=end]{justify-content:flex-end}.mat-mdc-dialog-actions .mat-button-base+.mat-button-base,.mat-mdc-dialog-actions .mat-mdc-button-base+.mat-mdc-button-base{margin-left:8px}[dir=rtl] .mat-mdc-dialog-actions .mat-button-base+.mat-button-base,[dir=rtl] .mat-mdc-dialog-actions .mat-mdc-button-base+.mat-mdc-button-base{margin-left:0;margin-right:8px}"] }]
        }], ctorParameters: () => [{ type: i0.ElementRef }, { type: i1.FocusTrapFactory }, { type: undefined, decorators: [{
                    type: Optional
                }, {
                    type: Inject,
                    args: [DOCUMENT]
                }] }, { type: i2.MatDialogConfig }, { type: i1.InteractivityChecker }, { type: i0.NgZone }, { type: i3.OverlayRef }, { type: undefined, decorators: [{
                    type: Optional
                }, {
                    type: Inject,
                    args: [ANIMATION_MODULE_TYPE]
                }] }, { type: i1.FocusMonitor }] });
const TRANSITION_DURATION_PROPERTY = '--mat-dialog-transition-duration';
// TODO(mmalerba): Remove this function after animation durations are required
//  to be numbers.
/**
 * Converts a CSS time string to a number in ms. If the given time is already a
 * number, it is assumed to be in ms.
 */
function parseCssTime(time) {
    if (time == null) {
        return null;
    }
    if (typeof time === 'number') {
        return time;
    }
    if (time.endsWith('ms')) {
        return coerceNumberProperty(time.substring(0, time.length - 2));
    }
    if (time.endsWith('s')) {
        return coerceNumberProperty(time.substring(0, time.length - 1)) * 1000;
    }
    if (time === '0') {
        return 0;
    }
    return null; // anything else is invalid.
}
//# sourceMappingURL=data:application/json;base64,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