/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { removeListItem } from '../util';
import { AbstractControl, isOptionsObj, pickAsyncValidators, pickValidators } from './abstract_model';
function isFormControlState(formState) {
    return typeof formState === 'object' && formState !== null &&
        Object.keys(formState).length === 2 && 'value' in formState && 'disabled' in formState;
}
export const FormControl = (class FormControl extends AbstractControl {
    constructor(
    // formState and defaultValue will only be null if T is nullable
    formState = null, validatorOrOpts, asyncValidator) {
        super(pickValidators(validatorOrOpts), pickAsyncValidators(asyncValidator, validatorOrOpts));
        /** @publicApi */
        this.defaultValue = null;
        /** @internal */
        this._onChange = [];
        /** @internal */
        this._pendingChange = false;
        this._applyFormState(formState);
        this._setUpdateStrategy(validatorOrOpts);
        this._initObservables();
        this.updateValueAndValidity({
            onlySelf: true,
            // If `asyncValidator` is present, it will trigger control status change from `PENDING` to
            // `VALID` or `INVALID`.
            // The status should be broadcasted via the `statusChanges` observable, so we set
            // `emitEvent` to `true` to allow that during the control creation process.
            emitEvent: !!this.asyncValidator
        });
        if (isOptionsObj(validatorOrOpts) &&
            (validatorOrOpts.nonNullable || validatorOrOpts.initialValueIsDefault)) {
            if (isFormControlState(formState)) {
                this.defaultValue = formState.value;
            }
            else {
                this.defaultValue = formState;
            }
        }
    }
    setValue(value, options = {}) {
        this.value = this._pendingValue = value;
        if (this._onChange.length && options.emitModelToViewChange !== false) {
            this._onChange.forEach((changeFn) => changeFn(this.value, options.emitViewToModelChange !== false));
        }
        this.updateValueAndValidity(options);
    }
    patchValue(value, options = {}) {
        this.setValue(value, options);
    }
    reset(formState = this.defaultValue, options = {}) {
        this._applyFormState(formState);
        this.markAsPristine(options);
        this.markAsUntouched(options);
        this.setValue(this.value, options);
        this._pendingChange = false;
    }
    /**  @internal */
    _updateValue() { }
    /**  @internal */
    _anyControls(condition) {
        return false;
    }
    /**  @internal */
    _allControlsDisabled() {
        return this.disabled;
    }
    registerOnChange(fn) {
        this._onChange.push(fn);
    }
    /** @internal */
    _unregisterOnChange(fn) {
        removeListItem(this._onChange, fn);
    }
    registerOnDisabledChange(fn) {
        this._onDisabledChange.push(fn);
    }
    /** @internal */
    _unregisterOnDisabledChange(fn) {
        removeListItem(this._onDisabledChange, fn);
    }
    /** @internal */
    _forEachChild(cb) { }
    /** @internal */
    _syncPendingControls() {
        if (this.updateOn === 'submit') {
            if (this._pendingDirty)
                this.markAsDirty();
            if (this._pendingTouched)
                this.markAsTouched();
            if (this._pendingChange) {
                this.setValue(this._pendingValue, { onlySelf: true, emitModelToViewChange: false });
                return true;
            }
        }
        return false;
    }
    _applyFormState(formState) {
        if (isFormControlState(formState)) {
            this.value = this._pendingValue = formState.value;
            formState.disabled ? this.disable({ onlySelf: true, emitEvent: false }) :
                this.enable({ onlySelf: true, emitEvent: false });
        }
        else {
            this.value = this._pendingValue = formState;
        }
    }
});
export const UntypedFormControl = FormControl;
/**
 * @description
 * Asserts that the given control is an instance of `FormControl`
 *
 * @publicApi
 */
export const isFormControl = (control) => control instanceof FormControl;
//# sourceMappingURL=data:application/json;base64,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