/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { REACTIVE_NODE } from '@angular/core/primitives/signals';
import { REACTIVE_TEMPLATE_CONSUMER } from './interfaces/view';
import { markAncestorsForTraversal } from './util/view_utils';
let freeConsumers = [];
/**
 * Create a new template consumer pointing at the specified LView.
 * Sometimes, a previously created consumer may be reused, in order to save on allocations. In that
 * case, the LView will be updated.
 */
export function getOrBorrowReactiveLViewConsumer(lView) {
    return lView[REACTIVE_TEMPLATE_CONSUMER] ?? borrowReactiveLViewConsumer(lView);
}
function borrowReactiveLViewConsumer(lView) {
    const consumer = freeConsumers.pop() ?? Object.create(REACTIVE_LVIEW_CONSUMER_NODE);
    consumer.lView = lView;
    return consumer;
}
export function maybeReturnReactiveLViewConsumer(consumer) {
    if (consumer.lView[REACTIVE_TEMPLATE_CONSUMER] === consumer) {
        // The consumer got committed.
        return;
    }
    consumer.lView = null;
    freeConsumers.push(consumer);
}
const REACTIVE_LVIEW_CONSUMER_NODE = {
    ...REACTIVE_NODE,
    consumerIsAlwaysLive: true,
    consumerMarkedDirty: (node) => {
        markAncestorsForTraversal(node.lView);
    },
    consumerOnSignalRead() {
        this.lView[REACTIVE_TEMPLATE_CONSUMER] = this;
    },
};
//# sourceMappingURL=data:application/json;base64,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