/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { resolveForwardRef } from '../../di';
import { RuntimeError } from '../../errors';
import { assertEqual } from '../../util/assert';
import { EMPTY_OBJ } from '../../util/empty';
import { getComponentDef, getDirectiveDef } from '../definition';
/**
 * This feature adds the host directives behavior to a directive definition by patching a
 * function onto it. The expectation is that the runtime will invoke the function during
 * directive matching.
 *
 * For example:
 * ```ts
 * class ComponentWithHostDirective {
 *   static ɵcmp = defineComponent({
 *    type: ComponentWithHostDirective,
 *    features: [ɵɵHostDirectivesFeature([
 *      SimpleHostDirective,
 *      {directive: AdvancedHostDirective, inputs: ['foo: alias'], outputs: ['bar']},
 *    ])]
 *  });
 * }
 * ```
 *
 * @codeGenApi
 */
export function ɵɵHostDirectivesFeature(rawHostDirectives) {
    const feature = (definition) => {
        const resolved = (Array.isArray(rawHostDirectives) ? rawHostDirectives : rawHostDirectives()).map(dir => {
            return typeof dir === 'function' ?
                { directive: resolveForwardRef(dir), inputs: EMPTY_OBJ, outputs: EMPTY_OBJ } :
                {
                    directive: resolveForwardRef(dir.directive),
                    inputs: bindingArrayToMap(dir.inputs),
                    outputs: bindingArrayToMap(dir.outputs)
                };
        });
        if (definition.hostDirectives === null) {
            definition.findHostDirectiveDefs = findHostDirectiveDefs;
            definition.hostDirectives = resolved;
        }
        else {
            definition.hostDirectives.unshift(...resolved);
        }
    };
    feature.ngInherit = true;
    return feature;
}
function findHostDirectiveDefs(currentDef, matchedDefs, hostDirectiveDefs) {
    if (currentDef.hostDirectives !== null) {
        for (const hostDirectiveConfig of currentDef.hostDirectives) {
            const hostDirectiveDef = getDirectiveDef(hostDirectiveConfig.directive);
            if (typeof ngDevMode === 'undefined' || ngDevMode) {
                validateHostDirective(hostDirectiveConfig, hostDirectiveDef);
            }
            // We need to patch the `declaredInputs` so that
            // `ngOnChanges` can map the properties correctly.
            patchDeclaredInputs(hostDirectiveDef.declaredInputs, hostDirectiveConfig.inputs);
            // Host directives execute before the host so that its host bindings can be overwritten.
            findHostDirectiveDefs(hostDirectiveDef, matchedDefs, hostDirectiveDefs);
            hostDirectiveDefs.set(hostDirectiveDef, hostDirectiveConfig);
            matchedDefs.push(hostDirectiveDef);
        }
    }
}
/**
 * Converts an array in the form of `['publicName', 'alias', 'otherPublicName', 'otherAlias']` into
 * a map in the form of `{publicName: 'alias', otherPublicName: 'otherAlias'}`.
 */
function bindingArrayToMap(bindings) {
    if (bindings === undefined || bindings.length === 0) {
        return EMPTY_OBJ;
    }
    const result = {};
    for (let i = 0; i < bindings.length; i += 2) {
        result[bindings[i]] = bindings[i + 1];
    }
    return result;
}
/**
 * `ngOnChanges` has some leftover legacy ViewEngine behavior where the keys inside the
 * `SimpleChanges` event refer to the *declared* name of the input, not its public name or its
 * minified name. E.g. in `@Input('alias') foo: string`, the name in the `SimpleChanges` object
 * will always be `foo`, and not `alias` or the minified name of `foo` in apps using property
 * minification.
 *
 * This is achieved through the `DirectiveDef.declaredInputs` map that is constructed when the
 * definition is declared. When a property is written to the directive instance, the
 * `NgOnChangesFeature` will try to remap the property name being written to using the
 * `declaredInputs`.
 *
 * Since the host directive input remapping happens during directive matching, `declaredInputs`
 * won't contain the new alias that the input is available under. This function addresses the
 * issue by patching the host directive aliases to the `declaredInputs`. There is *not* a risk of
 * this patching accidentally introducing new inputs to the host directive, because `declaredInputs`
 * is used *only* by the `NgOnChangesFeature` when determining what name is used in the
 * `SimpleChanges` object which won't be reached if an input doesn't exist.
 */
function patchDeclaredInputs(declaredInputs, exposedInputs) {
    for (const publicName in exposedInputs) {
        if (exposedInputs.hasOwnProperty(publicName)) {
            const remappedPublicName = exposedInputs[publicName];
            const privateName = declaredInputs[publicName];
            // We *technically* shouldn't be able to hit this case because we can't have multiple
            // inputs on the same property and we have validations against conflicting aliases in
            // `validateMappings`. If we somehow did, it would lead to `ngOnChanges` being invoked
            // with the wrong name so we have a non-user-friendly assertion here just in case.
            if ((typeof ngDevMode === 'undefined' || ngDevMode) &&
                declaredInputs.hasOwnProperty(remappedPublicName)) {
                assertEqual(declaredInputs[remappedPublicName], declaredInputs[publicName], `Conflicting host directive input alias ${publicName}.`);
            }
            declaredInputs[remappedPublicName] = privateName;
        }
    }
}
/**
 * Verifies that the host directive has been configured correctly.
 * @param hostDirectiveConfig Host directive configuration object.
 * @param directiveDef Directive definition of the host directive.
 */
function validateHostDirective(hostDirectiveConfig, directiveDef) {
    const type = hostDirectiveConfig.directive;
    if (directiveDef === null) {
        if (getComponentDef(type) !== null) {
            throw new RuntimeError(310 /* RuntimeErrorCode.HOST_DIRECTIVE_COMPONENT */, `Host directive ${type.name} cannot be a component.`);
        }
        throw new RuntimeError(307 /* RuntimeErrorCode.HOST_DIRECTIVE_UNRESOLVABLE */, `Could not resolve metadata for host directive ${type.name}. ` +
            `Make sure that the ${type.name} class is annotated with an @Directive decorator.`);
    }
    if (!directiveDef.standalone) {
        throw new RuntimeError(308 /* RuntimeErrorCode.HOST_DIRECTIVE_NOT_STANDALONE */, `Host directive ${directiveDef.type.name} must be standalone.`);
    }
    validateMappings('input', directiveDef, hostDirectiveConfig.inputs);
    validateMappings('output', directiveDef, hostDirectiveConfig.outputs);
}
/**
 * Checks that the host directive inputs/outputs configuration is valid.
 * @param bindingType Kind of binding that is being validated. Used in the error message.
 * @param def Definition of the host directive that is being validated against.
 * @param hostDirectiveBindings Host directive mapping object that shold be validated.
 */
function validateMappings(bindingType, def, hostDirectiveBindings) {
    const className = def.type.name;
    const bindings = bindingType === 'input' ? def.inputs : def.outputs;
    for (const publicName in hostDirectiveBindings) {
        if (hostDirectiveBindings.hasOwnProperty(publicName)) {
            if (!bindings.hasOwnProperty(publicName)) {
                throw new RuntimeError(311 /* RuntimeErrorCode.HOST_DIRECTIVE_UNDEFINED_BINDING */, `Directive ${className} does not have an ${bindingType} with a public name of ${publicName}.`);
            }
            const remappedPublicName = hostDirectiveBindings[publicName];
            if (bindings.hasOwnProperty(remappedPublicName) && remappedPublicName !== publicName &&
                bindings[remappedPublicName] !== publicName) {
                throw new RuntimeError(312 /* RuntimeErrorCode.HOST_DIRECTIVE_CONFLICTING_ALIAS */, `Cannot alias ${bindingType} ${publicName} of host directive ${className} to ${remappedPublicName}, because it already has a different ${bindingType} with the same public name.`);
            }
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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