/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ChangeDetectionStrategy } from '../change_detection/constants';
import { formatRuntimeError } from '../errors';
import { ViewEncapsulation } from '../metadata/view';
import { noSideEffects } from '../util/closure';
import { EMPTY_ARRAY, EMPTY_OBJ } from '../util/empty';
import { initNgDevMode } from '../util/ng_dev_mode';
import { stringify } from '../util/stringify';
import { NG_COMP_DEF, NG_DIR_DEF, NG_MOD_DEF, NG_PIPE_DEF } from './fields';
import { stringifyCSSSelectorList } from './node_selector_matcher';
/**
 * Create a component definition object.
 *
 *
 * # Example
 * ```
 * class MyComponent {
 *   // Generated by Angular Template Compiler
 *   // [Symbol] syntax will not be supported by TypeScript until v2.7
 *   static ɵcmp = defineComponent({
 *     ...
 *   });
 * }
 * ```
 * @codeGenApi
 */
export function ɵɵdefineComponent(componentDefinition) {
    return noSideEffects(() => {
        // Initialize ngDevMode. This must be the first statement in ɵɵdefineComponent.
        // See the `initNgDevMode` docstring for more information.
        (typeof ngDevMode === 'undefined' || ngDevMode) && initNgDevMode();
        const baseDef = getNgDirectiveDef(componentDefinition);
        const def = {
            ...baseDef,
            decls: componentDefinition.decls,
            vars: componentDefinition.vars,
            template: componentDefinition.template,
            consts: componentDefinition.consts || null,
            ngContentSelectors: componentDefinition.ngContentSelectors,
            onPush: componentDefinition.changeDetection === ChangeDetectionStrategy.OnPush,
            directiveDefs: null,
            pipeDefs: null,
            dependencies: baseDef.standalone && componentDefinition.dependencies || null,
            getStandaloneInjector: null,
            signals: componentDefinition.signals ?? false,
            data: componentDefinition.data || {},
            encapsulation: componentDefinition.encapsulation || ViewEncapsulation.Emulated,
            styles: componentDefinition.styles || EMPTY_ARRAY,
            _: null,
            schemas: componentDefinition.schemas || null,
            tView: null,
            id: '',
        };
        initFeatures(def);
        const dependencies = componentDefinition.dependencies;
        def.directiveDefs = extractDefListOrFactory(dependencies, /* pipeDef */ false);
        def.pipeDefs = extractDefListOrFactory(dependencies, /* pipeDef */ true);
        def.id = getComponentId(def);
        return def;
    });
}
export function extractDirectiveDef(type) {
    return getComponentDef(type) || getDirectiveDef(type);
}
function nonNull(value) {
    return value !== null;
}
/**
 * @codeGenApi
 */
export function ɵɵdefineNgModule(def) {
    return noSideEffects(() => {
        const res = {
            type: def.type,
            bootstrap: def.bootstrap || EMPTY_ARRAY,
            declarations: def.declarations || EMPTY_ARRAY,
            imports: def.imports || EMPTY_ARRAY,
            exports: def.exports || EMPTY_ARRAY,
            transitiveCompileScopes: null,
            schemas: def.schemas || null,
            id: def.id || null,
        };
        return res;
    });
}
/**
 * Inverts an inputs or outputs lookup such that the keys, which were the
 * minified keys, are part of the values, and the values are parsed so that
 * the publicName of the property is the new key
 *
 * e.g. for
 *
 * ```
 * class Comp {
 *   @Input()
 *   propName1: string;
 *
 *   @Input('publicName2')
 *   declaredPropName2: number;
 * }
 * ```
 *
 * will be serialized as
 *
 * ```
 * {
 *   propName1: 'propName1',
 *   declaredPropName2: ['publicName2', 'declaredPropName2'],
 * }
 * ```
 *
 * which is than translated by the minifier as:
 *
 * ```
 * {
 *   minifiedPropName1: 'propName1',
 *   minifiedPropName2: ['publicName2', 'declaredPropName2'],
 * }
 * ```
 *
 * becomes: (public name => minifiedName)
 *
 * ```
 * {
 *  'propName1': 'minifiedPropName1',
 *  'publicName2': 'minifiedPropName2',
 * }
 * ```
 *
 * Optionally the function can take `secondary` which will result in: (public name => declared name)
 *
 * ```
 * {
 *  'propName1': 'propName1',
 *  'publicName2': 'declaredPropName2',
 * }
 * ```
 *

 */
function invertObject(obj, secondary) {
    if (obj == null)
        return EMPTY_OBJ;
    const newLookup = {};
    for (const minifiedKey in obj) {
        if (obj.hasOwnProperty(minifiedKey)) {
            let publicName = obj[minifiedKey];
            let declaredName = publicName;
            if (Array.isArray(publicName)) {
                declaredName = publicName[1];
                publicName = publicName[0];
            }
            newLookup[publicName] = minifiedKey;
            if (secondary) {
                (secondary[publicName] = declaredName);
            }
        }
    }
    return newLookup;
}
/**
 * Create a directive definition object.
 *
 * # Example
 * ```ts
 * class MyDirective {
 *   // Generated by Angular Template Compiler
 *   // [Symbol] syntax will not be supported by TypeScript until v2.7
 *   static ɵdir = ɵɵdefineDirective({
 *     ...
 *   });
 * }
 * ```
 *
 * @codeGenApi
 */
export function ɵɵdefineDirective(directiveDefinition) {
    return noSideEffects(() => {
        const def = getNgDirectiveDef(directiveDefinition);
        initFeatures(def);
        return def;
    });
}
/**
 * Create a pipe definition object.
 *
 * # Example
 * ```
 * class MyPipe implements PipeTransform {
 *   // Generated by Angular Template Compiler
 *   static ɵpipe = definePipe({
 *     ...
 *   });
 * }
 * ```
 * @param pipeDef Pipe definition generated by the compiler
 *
 * @codeGenApi
 */
export function ɵɵdefinePipe(pipeDef) {
    return {
        type: pipeDef.type,
        name: pipeDef.name,
        factory: null,
        pure: pipeDef.pure !== false,
        standalone: pipeDef.standalone === true,
        onDestroy: pipeDef.type.prototype.ngOnDestroy || null
    };
}
/**
 * The following getter methods retrieve the definition from the type. Currently the retrieval
 * honors inheritance, but in the future we may change the rule to require that definitions are
 * explicit. This would require some sort of migration strategy.
 */
export function getComponentDef(type) {
    return type[NG_COMP_DEF] || null;
}
export function getDirectiveDef(type) {
    return type[NG_DIR_DEF] || null;
}
export function getPipeDef(type) {
    return type[NG_PIPE_DEF] || null;
}
/**
 * Checks whether a given Component, Directive or Pipe is marked as standalone.
 * This will return false if passed anything other than a Component, Directive, or Pipe class
 * See [this guide](/guide/standalone-components) for additional information:
 *
 * @param type A reference to a Component, Directive or Pipe.
 * @publicApi
 */
export function isStandalone(type) {
    const def = getComponentDef(type) || getDirectiveDef(type) || getPipeDef(type);
    return def !== null ? def.standalone : false;
}
export function getNgModuleDef(type, throwNotFound) {
    const ngModuleDef = type[NG_MOD_DEF] || null;
    if (!ngModuleDef && throwNotFound === true) {
        throw new Error(`Type ${stringify(type)} does not have 'ɵmod' property.`);
    }
    return ngModuleDef;
}
function getNgDirectiveDef(directiveDefinition) {
    const declaredInputs = {};
    return {
        type: directiveDefinition.type,
        providersResolver: null,
        factory: null,
        hostBindings: directiveDefinition.hostBindings || null,
        hostVars: directiveDefinition.hostVars || 0,
        hostAttrs: directiveDefinition.hostAttrs || null,
        contentQueries: directiveDefinition.contentQueries || null,
        declaredInputs,
        inputTransforms: null,
        inputConfig: directiveDefinition.inputs || EMPTY_OBJ,
        exportAs: directiveDefinition.exportAs || null,
        standalone: directiveDefinition.standalone === true,
        signals: directiveDefinition.signals === true,
        selectors: directiveDefinition.selectors || EMPTY_ARRAY,
        viewQuery: directiveDefinition.viewQuery || null,
        features: directiveDefinition.features || null,
        setInput: null,
        findHostDirectiveDefs: null,
        hostDirectives: null,
        inputs: invertObject(directiveDefinition.inputs, declaredInputs),
        outputs: invertObject(directiveDefinition.outputs),
        debugInfo: null,
    };
}
function initFeatures(definition) {
    definition.features?.forEach((fn) => fn(definition));
}
export function extractDefListOrFactory(dependencies, pipeDef) {
    if (!dependencies) {
        return null;
    }
    const defExtractor = pipeDef ? getPipeDef : extractDirectiveDef;
    return () => (typeof dependencies === 'function' ? dependencies() : dependencies)
        .map(dep => defExtractor(dep))
        .filter(nonNull);
}
/**
 * A map that contains the generated component IDs and type.
 */
export const GENERATED_COMP_IDS = new Map();
/**
 * A method can returns a component ID from the component definition using a variant of DJB2 hash
 * algorithm.
 */
function getComponentId(componentDef) {
    let hash = 0;
    // We cannot rely solely on the component selector as the same selector can be used in different
    // modules.
    //
    // `componentDef.style` is not used, due to it causing inconsistencies. Ex: when server
    // component styles has no sourcemaps and browsers do.
    //
    // Example:
    // https://github.com/angular/components/blob/d9f82c8f95309e77a6d82fd574c65871e91354c2/src/material/core/option/option.ts#L248
    // https://github.com/angular/components/blob/285f46dc2b4c5b127d356cb7c4714b221f03ce50/src/material/legacy-core/option/option.ts#L32
    const hashSelectors = [
        componentDef.selectors,
        componentDef.ngContentSelectors,
        componentDef.hostVars,
        componentDef.hostAttrs,
        componentDef.consts,
        componentDef.vars,
        componentDef.decls,
        componentDef.encapsulation,
        componentDef.standalone,
        componentDef.signals,
        componentDef.exportAs,
        JSON.stringify(componentDef.inputs),
        JSON.stringify(componentDef.outputs),
        // We cannot use 'componentDef.type.name' as the name of the symbol will change and will not
        // match in the server and browser bundles.
        Object.getOwnPropertyNames(componentDef.type.prototype),
        !!componentDef.contentQueries,
        !!componentDef.viewQuery,
    ].join('|');
    for (const char of hashSelectors) {
        hash = Math.imul(31, hash) + char.charCodeAt(0) << 0;
    }
    // Force positive number hash.
    // 2147483647 = equivalent of Integer.MAX_VALUE.
    hash += 2147483647 + 1;
    const compId = 'c' + hash;
    if (typeof ngDevMode === 'undefined' || ngDevMode) {
        if (GENERATED_COMP_IDS.has(compId)) {
            const previousCompDefType = GENERATED_COMP_IDS.get(compId);
            if (previousCompDefType !== componentDef.type) {
                console.warn(formatRuntimeError(-912 /* RuntimeErrorCode.COMPONENT_ID_COLLISION */, `Component ID generation collision detected. Components '${previousCompDefType.name}' and '${componentDef.type.name}' with selector '${stringifyCSSSelectorList(componentDef
                    .selectors)}' generated the same component ID. To fix this, you can change the selector of one of those components or add an extra host attribute to force a different ID.`));
            }
        }
        else {
            GENERATED_COMP_IDS.set(compId, componentDef.type);
        }
    }
    return compId;
}
//# sourceMappingURL=data:application/json;base64,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