/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { defaultEquals } from './equality';
import { throwInvalidWriteToSignalError } from './errors';
import { producerAccessed, producerIncrementEpoch, producerNotifyConsumers, producerUpdatesAllowed, REACTIVE_NODE, SIGNAL } from './graph';
/**
 * If set, called after `WritableSignal`s are updated.
 *
 * This hook can be used to achieve various effects, such as running effects synchronously as part
 * of setting a signal.
 */
let postSignalSetFn = null;
/**
 * Create a `Signal` that can be set or updated directly.
 */
export function createSignal(initialValue) {
    const node = Object.create(SIGNAL_NODE);
    node.value = initialValue;
    const getter = (() => {
        producerAccessed(node);
        return node.value;
    });
    getter[SIGNAL] = node;
    return getter;
}
export function setPostSignalSetFn(fn) {
    const prev = postSignalSetFn;
    postSignalSetFn = fn;
    return prev;
}
export function signalGetFn() {
    producerAccessed(this);
    return this.value;
}
export function signalSetFn(node, newValue) {
    if (!producerUpdatesAllowed()) {
        throwInvalidWriteToSignalError();
    }
    const value = node.value;
    if (Object.is(value, newValue)) {
        if (typeof ngDevMode !== 'undefined' && ngDevMode && !node.equal(value, newValue)) {
            console.warn('Signal value equality implementations should always return `true` for' +
                ' values that are the same according to `Object.is` but returned `false` instead.');
        }
    }
    else if (!node.equal(value, newValue)) {
        node.value = newValue;
        signalValueChanged(node);
    }
}
export function signalUpdateFn(node, updater) {
    if (!producerUpdatesAllowed()) {
        throwInvalidWriteToSignalError();
    }
    signalSetFn(node, updater(node.value));
}
export function signalMutateFn(node, mutator) {
    if (!producerUpdatesAllowed()) {
        throwInvalidWriteToSignalError();
    }
    // Mutate bypasses equality checks as it's by definition changing the value.
    mutator(node.value);
    signalValueChanged(node);
}
// Note: Using an IIFE here to ensure that the spread assignment is not considered
// a side-effect, ending up preserving `COMPUTED_NODE` and `REACTIVE_NODE`.
// TODO: remove when https://github.com/evanw/esbuild/issues/3392 is resolved.
const SIGNAL_NODE = /* @__PURE__ */ (() => {
    return {
        ...REACTIVE_NODE,
        equal: defaultEquals,
        value: undefined,
    };
})();
function signalValueChanged(node) {
    node.version++;
    producerIncrementEpoch();
    producerNotifyConsumers(node);
    postSignalSetFn?.();
}
//# sourceMappingURL=data:application/json;base64,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