/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import * as o from '../../../../output/output_ast';
import * as ir from '../../ir';
/**
 * Optimize variables declared and used in the IR.
 *
 * Variables are eagerly generated by pipeline stages for all possible values that could be
 * referenced. This stage processes the list of declared variables and all variable usages,
 * and optimizes where possible. It performs 3 main optimizations:
 *
 *   * It transforms variable declarations to side effectful expressions when the
 *     variable is not used, but its initializer has global effects which other
 *     operations rely upon.
 *   * It removes variable declarations if those variables are not referenced and
 *     either they do not have global effects, or nothing relies on them.
 *   * It inlines variable declarations when those variables are only used once
 *     and the inlining is semantically safe.
 *
 * To guarantee correctness, analysis of "fences" in the instruction lists is used to determine
 * which optimizations are safe to perform.
 */
export function optimizeVariables(job) {
    for (const unit of job.units) {
        inlineAlwaysInlineVariables(unit.create);
        inlineAlwaysInlineVariables(unit.update);
        for (const op of unit.create) {
            if (op.kind === ir.OpKind.Listener) {
                inlineAlwaysInlineVariables(op.handlerOps);
            }
        }
        optimizeVariablesInOpList(unit.create, job.compatibility);
        optimizeVariablesInOpList(unit.update, job.compatibility);
        for (const op of unit.create) {
            if (op.kind === ir.OpKind.Listener) {
                optimizeVariablesInOpList(op.handlerOps, job.compatibility);
            }
        }
    }
}
/**
 * A [fence](https://en.wikipedia.org/wiki/Memory_barrier) flag for an expression which indicates
 * how that expression can be optimized in relation to other expressions or instructions.
 *
 * `Fence`s are a bitfield, so multiple flags may be set on a single expression.
 */
var Fence;
(function (Fence) {
    /**
     * Empty flag (no fence exists).
     */
    Fence[Fence["None"] = 0] = "None";
    /**
     * A context read fence, meaning that the expression in question reads from the "current view"
     * context of the runtime.
     */
    Fence[Fence["ViewContextRead"] = 1] = "ViewContextRead";
    /**
     * A context write fence, meaning that the expression in question writes to the "current view"
     * context of the runtime.
     *
     * Note that all `ContextWrite` fences are implicitly `ContextRead` fences as operations which
     * change the view context do so based on the current one.
     */
    Fence[Fence["ViewContextWrite"] = 2] = "ViewContextWrite";
    /**
     * Indicates that a call is required for its side-effects, even if nothing reads its result.
     *
     * This is also true of `ViewContextWrite` operations **if** they are followed by a
     * `ViewContextRead`.
     */
    Fence[Fence["SideEffectful"] = 4] = "SideEffectful";
})(Fence || (Fence = {}));
function inlineAlwaysInlineVariables(ops) {
    const vars = new Map();
    for (const op of ops) {
        if (op.kind === ir.OpKind.Variable && op.flags & ir.VariableFlags.AlwaysInline) {
            ir.visitExpressionsInOp(op, expr => {
                if (ir.isIrExpression(expr) && fencesForIrExpression(expr) !== Fence.None) {
                    throw new Error(`AssertionError: A context-sensitive variable was marked AlwaysInline`);
                }
            });
            vars.set(op.xref, op);
        }
        ir.transformExpressionsInOp(op, expr => {
            if (expr instanceof ir.ReadVariableExpr && vars.has(expr.xref)) {
                const varOp = vars.get(expr.xref);
                // Inline by cloning, because we might inline into multiple places.
                return varOp.initializer.clone();
            }
            return expr;
        }, ir.VisitorContextFlag.None);
    }
    for (const op of vars.values()) {
        ir.OpList.remove(op);
    }
}
/**
 * Process a list of operations and optimize variables within that list.
 */
function optimizeVariablesInOpList(ops, compatibility) {
    const varDecls = new Map();
    const varUsages = new Map();
    // Track variables that are used outside of the immediate operation list. For example, within
    // `ListenerOp` handler operations of listeners in the current operation list.
    const varRemoteUsages = new Set();
    const opMap = new Map();
    // First, extract information about variables declared or used within the whole list.
    for (const op of ops) {
        if (op.kind === ir.OpKind.Variable) {
            if (varDecls.has(op.xref) || varUsages.has(op.xref)) {
                throw new Error(`Should not see two declarations of the same variable: ${op.xref}`);
            }
            varDecls.set(op.xref, op);
            varUsages.set(op.xref, 0);
        }
        opMap.set(op, collectOpInfo(op));
        countVariableUsages(op, varUsages, varRemoteUsages);
    }
    // The next step is to remove any variable declarations for variables that aren't used. The
    // variable initializer expressions may be side-effectful, so they may need to be retained as
    // expression statements.
    // Track whether we've seen an operation which reads from the view context yet. This is used to
    // determine whether a write to the view context in a variable initializer can be observed.
    let contextIsUsed = false;
    // Note that iteration through the list happens in reverse, which guarantees that we'll process
    // all reads of a variable prior to processing its declaration.
    for (const op of ops.reversed()) {
        const opInfo = opMap.get(op);
        if (op.kind === ir.OpKind.Variable && varUsages.get(op.xref) === 0) {
            // This variable is unused and can be removed. We might need to keep the initializer around,
            // though, if something depends on it running.
            if ((contextIsUsed && opInfo.fences & Fence.ViewContextWrite) ||
                (opInfo.fences & Fence.SideEffectful)) {
                // This variable initializer has a side effect which must be retained. Either:
                //  * it writes to the view context, and we know there is a future operation which depends
                //    on that write, or
                //  * it's an operation which is inherently side-effectful.
                // We can't remove the initializer, but we can remove the variable declaration itself and
                // replace it with a side-effectful statement.
                const stmtOp = ir.createStatementOp(op.initializer.toStmt());
                opMap.set(stmtOp, opInfo);
                ir.OpList.replace(op, stmtOp);
            }
            else {
                // It's safe to delete this entire variable declaration as nothing depends on it, even
                // side-effectfully. Note that doing this might make other variables unused. Since we're
                // iterating in reverse order, we should always be processing usages before declarations
                // and therefore by the time we get to a declaration, all removable usages will have been
                // removed.
                uncountVariableUsages(op, varUsages);
                ir.OpList.remove(op);
            }
            opMap.delete(op);
            varDecls.delete(op.xref);
            varUsages.delete(op.xref);
            continue;
        }
        // Does this operation depend on the view context?
        if (opInfo.fences & Fence.ViewContextRead) {
            contextIsUsed = true;
        }
    }
    // Next, inline any remaining variables with exactly one usage.
    const toInline = [];
    for (const [id, count] of varUsages) {
        const decl = varDecls.get(id);
        const varInfo = opMap.get(decl);
        // We can inline variables that:
        //  - are used exactly once, and
        //  - are not used remotely
        // OR
        //  - are marked for always inlining
        const isAlwaysInline = !!(decl.flags & ir.VariableFlags.AlwaysInline);
        if (count !== 1 || isAlwaysInline) {
            // We can't inline this variable as it's used more than once.
            continue;
        }
        if (varRemoteUsages.has(id)) {
            // This variable is used once, but across an operation boundary, so it can't be inlined.
            continue;
        }
        toInline.push(id);
    }
    let candidate;
    while (candidate = toInline.pop()) {
        // We will attempt to inline this variable. If inlining fails (due to fences for example),
        // no future operation will make inlining legal.
        const decl = varDecls.get(candidate);
        const varInfo = opMap.get(decl);
        const isAlwaysInline = !!(decl.flags & ir.VariableFlags.AlwaysInline);
        if (isAlwaysInline) {
            throw new Error(`AssertionError: Found an 'AlwaysInline' variable after the always inlining pass.`);
        }
        // Scan operations following the variable declaration and look for the point where that variable
        // is used. There should only be one usage given the precondition above.
        for (let targetOp = decl.next; targetOp.kind !== ir.OpKind.ListEnd; targetOp = targetOp.next) {
            const opInfo = opMap.get(targetOp);
            // Is the variable used in this operation?
            if (opInfo.variablesUsed.has(candidate)) {
                if (compatibility === ir.CompatibilityMode.TemplateDefinitionBuilder &&
                    !allowConservativeInlining(decl, targetOp)) {
                    // We're in conservative mode, and this variable is not eligible for inlining into the
                    // target operation in this mode.
                    break;
                }
                // Yes, try to inline it. Inlining may not be successful if fences in this operation before
                // the variable's usage cannot be safely crossed.
                if (tryInlineVariableInitializer(candidate, decl.initializer, targetOp, varInfo.fences)) {
                    // Inlining was successful! Update the tracking structures to reflect the inlined
                    // variable.
                    opInfo.variablesUsed.delete(candidate);
                    // Add all variables used in the variable's initializer to its new usage site.
                    for (const id of varInfo.variablesUsed) {
                        opInfo.variablesUsed.add(id);
                    }
                    // Merge fences in the variable's initializer into its new usage site.
                    opInfo.fences |= varInfo.fences;
                    // Delete tracking info related to the declaration.
                    varDecls.delete(candidate);
                    varUsages.delete(candidate);
                    opMap.delete(decl);
                    // And finally, delete the original declaration from the operation list.
                    ir.OpList.remove(decl);
                }
                // Whether inlining succeeded or failed, we're done processing this variable.
                break;
            }
            // If the variable is not used in this operation, then we'd need to inline across it. Check if
            // that's safe to do.
            if (!safeToInlinePastFences(opInfo.fences, varInfo.fences)) {
                // We can't safely inline this variable beyond this operation, so don't proceed with
                // inlining this variable.
                break;
            }
        }
    }
}
/**
 * Given an `ir.Expression`, returns the `Fence` flags for that expression type.
 */
function fencesForIrExpression(expr) {
    switch (expr.kind) {
        case ir.ExpressionKind.NextContext:
            return Fence.ViewContextRead | Fence.ViewContextWrite;
        case ir.ExpressionKind.RestoreView:
            return Fence.ViewContextRead | Fence.ViewContextWrite | Fence.SideEffectful;
        case ir.ExpressionKind.Reference:
            return Fence.ViewContextRead;
        default:
            return Fence.None;
    }
}
/**
 * Build the `OpInfo` structure for the given `op`. This performs two operations:
 *
 *  * It tracks which variables are used in the operation's expressions.
 *  * It rolls up fence flags for expressions within the operation.
 */
function collectOpInfo(op) {
    let fences = Fence.None;
    const variablesUsed = new Set();
    ir.visitExpressionsInOp(op, expr => {
        if (!ir.isIrExpression(expr)) {
            return;
        }
        switch (expr.kind) {
            case ir.ExpressionKind.ReadVariable:
                variablesUsed.add(expr.xref);
                break;
            default:
                fences |= fencesForIrExpression(expr);
        }
    });
    return { fences, variablesUsed };
}
/**
 * Count the number of usages of each variable, being careful to track whether those usages are
 * local or remote.
 */
function countVariableUsages(op, varUsages, varRemoteUsage) {
    ir.visitExpressionsInOp(op, (expr, flags) => {
        if (!ir.isIrExpression(expr)) {
            return;
        }
        if (expr.kind !== ir.ExpressionKind.ReadVariable) {
            return;
        }
        const count = varUsages.get(expr.xref);
        if (count === undefined) {
            // This variable is declared outside the current scope of optimization.
            return;
        }
        varUsages.set(expr.xref, count + 1);
        if (flags & ir.VisitorContextFlag.InChildOperation) {
            varRemoteUsage.add(expr.xref);
        }
    });
}
/**
 * Remove usages of a variable in `op` from the `varUsages` tracking.
 */
function uncountVariableUsages(op, varUsages) {
    ir.visitExpressionsInOp(op, expr => {
        if (!ir.isIrExpression(expr)) {
            return;
        }
        if (expr.kind !== ir.ExpressionKind.ReadVariable) {
            return;
        }
        const count = varUsages.get(expr.xref);
        if (count === undefined) {
            // This variable is declared outside the current scope of optimization.
            return;
        }
        else if (count === 0) {
            throw new Error(`Inaccurate variable count: ${expr.xref} - found another read but count is already 0`);
        }
        varUsages.set(expr.xref, count - 1);
    });
}
/**
 * Checks whether it's safe to inline a variable across a particular operation.
 *
 * @param fences the fences of the operation which the inlining will cross
 * @param declFences the fences of the variable being inlined.
 */
function safeToInlinePastFences(fences, declFences) {
    if (fences & Fence.ViewContextWrite) {
        // It's not safe to inline context reads across context writes.
        if (declFences & Fence.ViewContextRead) {
            return false;
        }
    }
    else if (fences & Fence.ViewContextRead) {
        // It's not safe to inline context writes across context reads.
        if (declFences & Fence.ViewContextWrite) {
            return false;
        }
    }
    return true;
}
/**
 * Attempt to inline the initializer of a variable into a target operation's expressions.
 *
 * This may or may not be safe to do. For example, the variable could be read following the
 * execution of an expression with fences that don't permit the variable to be inlined across them.
 */
function tryInlineVariableInitializer(id, initializer, target, declFences) {
    // We use `ir.transformExpressionsInOp` to walk the expressions and inline the variable if
    // possible. Since this operation is callback-based, once inlining succeeds or fails we can't
    // "stop" the expression processing, and have to keep track of whether inlining has succeeded or
    // is no longer allowed.
    let inlined = false;
    let inliningAllowed = true;
    ir.transformExpressionsInOp(target, (expr, flags) => {
        if (!ir.isIrExpression(expr)) {
            return expr;
        }
        if (inlined || !inliningAllowed) {
            // Either the inlining has already succeeded, or we've passed a fence that disallows inlining
            // at this point, so don't try.
            return expr;
        }
        else if ((flags & ir.VisitorContextFlag.InChildOperation) && (declFences & Fence.ViewContextRead)) {
            // We cannot inline variables that are sensitive to the current context across operation
            // boundaries.
            return expr;
        }
        switch (expr.kind) {
            case ir.ExpressionKind.ReadVariable:
                if (expr.xref === id) {
                    // This is the usage site of the variable. Since nothing has disallowed inlining, it's
                    // safe to inline the initializer here.
                    inlined = true;
                    return initializer;
                }
                break;
            default:
                // For other types of `ir.Expression`s, whether inlining is allowed depends on their fences.
                const exprFences = fencesForIrExpression(expr);
                inliningAllowed = inliningAllowed && safeToInlinePastFences(exprFences, declFences);
                break;
        }
        return expr;
    }, ir.VisitorContextFlag.None);
    return inlined;
}
/**
 * Determines whether inlining of `decl` should be allowed in "conservative" mode.
 *
 * In conservative mode, inlining behavior is limited to those operations which the
 * `TemplateDefinitionBuilder` supported, with the goal of producing equivalent output.
 */
function allowConservativeInlining(decl, target) {
    // TODO(alxhub): understand exactly how TemplateDefinitionBuilder approaches inlining, and record
    // that behavior here.
    switch (decl.variable.kind) {
        case ir.SemanticVariableKind.Identifier:
            if (decl.initializer instanceof o.ReadVarExpr && decl.initializer.name === 'ctx') {
                // Although TemplateDefinitionBuilder is cautious about inlining, we still want to do so
                // when the variable is the context, to imitate its behavior with aliases in control flow
                // blocks. This quirky behavior will become dead code once compatibility mode is no longer
                // supported.
                return true;
            }
            return false;
        case ir.SemanticVariableKind.Context:
            // Context can only be inlined into other variables.
            return target.kind === ir.OpKind.Variable;
        default:
            return true;
    }
}
//# sourceMappingURL=data:application/json;base64,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