/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import * as o from '../../../../output/output_ast';
import * as ir from '../../ir';
/**
 * Safe read expressions such as `a?.b` have different semantics in Angular templates as
 * compared to JavaScript. In particular, they default to `null` instead of `undefined`. This phase
 * finds all unresolved safe read expressions, and converts them into the appropriate output AST
 * reads, guarded by null checks. We generate temporaries as needed, to avoid re-evaluating the same
 * sub-expression multiple times.
 */
export function expandSafeReads(job) {
    for (const unit of job.units) {
        for (const op of unit.ops()) {
            ir.transformExpressionsInOp(op, e => safeTransform(e, { job }), ir.VisitorContextFlag.None);
            ir.transformExpressionsInOp(op, ternaryTransform, ir.VisitorContextFlag.None);
        }
    }
}
// A lookup set of all the expression kinds that require a temporary variable to be generated.
const requiresTemporary = [
    o.InvokeFunctionExpr, o.LiteralArrayExpr, o.LiteralMapExpr, ir.SafeInvokeFunctionExpr,
    ir.PipeBindingExpr
].map(e => e.constructor.name);
function needsTemporaryInSafeAccess(e) {
    // TODO: We probably want to use an expression visitor to recursively visit all descendents.
    // However, that would potentially do a lot of extra work (because it cannot short circuit), so we
    // implement the logic ourselves for now.
    if (e instanceof o.UnaryOperatorExpr) {
        return needsTemporaryInSafeAccess(e.expr);
    }
    else if (e instanceof o.BinaryOperatorExpr) {
        return needsTemporaryInSafeAccess(e.lhs) || needsTemporaryInSafeAccess(e.rhs);
    }
    else if (e instanceof o.ConditionalExpr) {
        if (e.falseCase && needsTemporaryInSafeAccess(e.falseCase))
            return true;
        return needsTemporaryInSafeAccess(e.condition) || needsTemporaryInSafeAccess(e.trueCase);
    }
    else if (e instanceof o.NotExpr) {
        return needsTemporaryInSafeAccess(e.condition);
    }
    else if (e instanceof ir.AssignTemporaryExpr) {
        return needsTemporaryInSafeAccess(e.expr);
    }
    else if (e instanceof o.ReadPropExpr) {
        return needsTemporaryInSafeAccess(e.receiver);
    }
    else if (e instanceof o.ReadKeyExpr) {
        return needsTemporaryInSafeAccess(e.receiver) || needsTemporaryInSafeAccess(e.index);
    }
    // TODO: Switch to a method which is exhaustive of newly added expression subtypes.
    return e instanceof o.InvokeFunctionExpr || e instanceof o.LiteralArrayExpr ||
        e instanceof o.LiteralMapExpr || e instanceof ir.SafeInvokeFunctionExpr ||
        e instanceof ir.PipeBindingExpr;
}
function temporariesIn(e) {
    const temporaries = new Set();
    // TODO: Although it's not currently supported by the transform helper, we should be able to
    // short-circuit exploring the tree to do less work. In particular, we don't have to penetrate
    // into the subexpressions of temporary assignments.
    ir.transformExpressionsInExpression(e, e => {
        if (e instanceof ir.AssignTemporaryExpr) {
            temporaries.add(e.xref);
        }
        return e;
    }, ir.VisitorContextFlag.None);
    return temporaries;
}
function eliminateTemporaryAssignments(e, tmps, ctx) {
    // TODO: We can be more efficient than the transform helper here. We don't need to visit any
    // descendents of temporary assignments.
    ir.transformExpressionsInExpression(e, e => {
        if (e instanceof ir.AssignTemporaryExpr && tmps.has(e.xref)) {
            const read = new ir.ReadTemporaryExpr(e.xref);
            // `TemplateDefinitionBuilder` has the (accidental?) behavior of generating assignments of
            // temporary variables to themselves. This happens because some subexpression that the
            // temporary refers to, possibly through nested temporaries, has a function call. We copy that
            // behavior here.
            return ctx.job.compatibility === ir.CompatibilityMode.TemplateDefinitionBuilder ?
                new ir.AssignTemporaryExpr(read, read.xref) :
                read;
        }
        return e;
    }, ir.VisitorContextFlag.None);
    return e;
}
/**
 * Creates a safe ternary guarded by the input expression, and with a body generated by the provided
 * callback on the input expression. Generates a temporary variable assignment if needed, and
 * deduplicates nested temporary assignments if needed.
 */
function safeTernaryWithTemporary(guard, body, ctx) {
    let result;
    if (needsTemporaryInSafeAccess(guard)) {
        const xref = ctx.job.allocateXrefId();
        result = [new ir.AssignTemporaryExpr(guard, xref), new ir.ReadTemporaryExpr(xref)];
    }
    else {
        result = [guard, guard.clone()];
        // Consider an expression like `a?.[b?.c()]?.d`. The `b?.c()` will be transformed first,
        // introducing a temporary assignment into the key. Then, as part of expanding the `?.d`. That
        // assignment will be duplicated into both the guard and expression sides. We de-duplicate it,
        // by transforming it from an assignment into a read on the expression side.
        eliminateTemporaryAssignments(result[1], temporariesIn(result[0]), ctx);
    }
    return new ir.SafeTernaryExpr(result[0], body(result[1]));
}
function isSafeAccessExpression(e) {
    return e instanceof ir.SafePropertyReadExpr || e instanceof ir.SafeKeyedReadExpr ||
        e instanceof ir.SafeInvokeFunctionExpr;
}
function isUnsafeAccessExpression(e) {
    return e instanceof o.ReadPropExpr || e instanceof o.ReadKeyExpr ||
        e instanceof o.InvokeFunctionExpr;
}
function isAccessExpression(e) {
    return isSafeAccessExpression(e) || isUnsafeAccessExpression(e);
}
function deepestSafeTernary(e) {
    if (isAccessExpression(e) && e.receiver instanceof ir.SafeTernaryExpr) {
        let st = e.receiver;
        while (st.expr instanceof ir.SafeTernaryExpr) {
            st = st.expr;
        }
        return st;
    }
    return null;
}
// TODO: When strict compatibility with TemplateDefinitionBuilder is not required, we can use `&&`
// instead to save some code size.
function safeTransform(e, ctx) {
    if (!isAccessExpression(e)) {
        return e;
    }
    const dst = deepestSafeTernary(e);
    if (dst) {
        if (e instanceof o.InvokeFunctionExpr) {
            dst.expr = dst.expr.callFn(e.args);
            return e.receiver;
        }
        if (e instanceof o.ReadPropExpr) {
            dst.expr = dst.expr.prop(e.name);
            return e.receiver;
        }
        if (e instanceof o.ReadKeyExpr) {
            dst.expr = dst.expr.key(e.index);
            return e.receiver;
        }
        if (e instanceof ir.SafeInvokeFunctionExpr) {
            dst.expr = safeTernaryWithTemporary(dst.expr, (r) => r.callFn(e.args), ctx);
            return e.receiver;
        }
        if (e instanceof ir.SafePropertyReadExpr) {
            dst.expr = safeTernaryWithTemporary(dst.expr, (r) => r.prop(e.name), ctx);
            return e.receiver;
        }
        if (e instanceof ir.SafeKeyedReadExpr) {
            dst.expr = safeTernaryWithTemporary(dst.expr, (r) => r.key(e.index), ctx);
            return e.receiver;
        }
    }
    else {
        if (e instanceof ir.SafeInvokeFunctionExpr) {
            return safeTernaryWithTemporary(e.receiver, (r) => r.callFn(e.args), ctx);
        }
        if (e instanceof ir.SafePropertyReadExpr) {
            return safeTernaryWithTemporary(e.receiver, (r) => r.prop(e.name), ctx);
        }
        if (e instanceof ir.SafeKeyedReadExpr) {
            return safeTernaryWithTemporary(e.receiver, (r) => r.key(e.index), ctx);
        }
    }
    return e;
}
function ternaryTransform(e) {
    if (!(e instanceof ir.SafeTernaryExpr)) {
        return e;
    }
    return new o.ConditionalExpr(new o.BinaryOperatorExpr(o.BinaryOperator.Equals, e.guard, o.NULL_EXPR), o.NULL_EXPR, e.expr);
}
//# sourceMappingURL=data:application/json;base64,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