/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import * as o from '../../../../../output/output_ast';
import { Namespace, OpKind } from '../enums';
import { SlotHandle } from '../handle';
import { OpList } from '../operations';
import { TRAIT_CONSUMES_SLOT } from '../traits';
import { NEW_OP } from './shared';
/**
 * The set of OpKinds that represent the creation of an element or container
 */
const elementContainerOpKinds = new Set([
    OpKind.Element, OpKind.ElementStart, OpKind.Container, OpKind.ContainerStart, OpKind.Template,
    OpKind.RepeaterCreate
]);
/**
 * Checks whether the given operation represents the creation of an element or container.
 */
export function isElementOrContainerOp(op) {
    return elementContainerOpKinds.has(op.kind);
}
/**
 * Create an `ElementStartOp`.
 */
export function createElementStartOp(tag, xref, namespace, i18nPlaceholder, sourceSpan) {
    return {
        kind: OpKind.ElementStart,
        xref,
        tag,
        handle: new SlotHandle(),
        attributes: null,
        localRefs: [],
        nonBindable: false,
        namespace,
        i18nPlaceholder,
        sourceSpan,
        ...TRAIT_CONSUMES_SLOT,
        ...NEW_OP,
    };
}
/**
 * Create a `TemplateOp`.
 */
export function createTemplateOp(xref, tag, functionNameSuffix, namespace, i18nPlaceholder, sourceSpan) {
    return {
        kind: OpKind.Template,
        xref,
        attributes: null,
        tag,
        handle: new SlotHandle(),
        functionNameSuffix,
        decls: null,
        vars: null,
        localRefs: [],
        nonBindable: false,
        namespace,
        i18nPlaceholder,
        sourceSpan,
        ...TRAIT_CONSUMES_SLOT,
        ...NEW_OP,
    };
}
export function createRepeaterCreateOp(primaryView, emptyView, tag, track, varNames, sourceSpan) {
    return {
        kind: OpKind.RepeaterCreate,
        attributes: null,
        xref: primaryView,
        handle: new SlotHandle(),
        emptyView,
        track,
        trackByFn: null,
        tag,
        functionNameSuffix: 'For',
        namespace: Namespace.HTML,
        nonBindable: false,
        localRefs: [],
        decls: null,
        vars: null,
        varNames,
        usesComponentInstance: false,
        sourceSpan,
        ...TRAIT_CONSUMES_SLOT,
        ...NEW_OP,
        numSlotsUsed: emptyView === null ? 2 : 3,
    };
}
/**
 * Create an `ElementEndOp`.
 */
export function createElementEndOp(xref, sourceSpan) {
    return {
        kind: OpKind.ElementEnd,
        xref,
        sourceSpan,
        ...NEW_OP,
    };
}
export function createDisableBindingsOp(xref) {
    return {
        kind: OpKind.DisableBindings,
        xref,
        ...NEW_OP,
    };
}
export function createEnableBindingsOp(xref) {
    return {
        kind: OpKind.EnableBindings,
        xref,
        ...NEW_OP,
    };
}
/**
 * Create a `TextOp`.
 */
export function createTextOp(xref, initialValue, sourceSpan) {
    return {
        kind: OpKind.Text,
        xref,
        handle: new SlotHandle(),
        initialValue,
        sourceSpan,
        ...TRAIT_CONSUMES_SLOT,
        ...NEW_OP,
    };
}
/**
 * Create a `ListenerOp`. Host bindings reuse all the listener logic.
 */
export function createListenerOp(target, targetSlot, name, tag, animationPhase, hostListener, sourceSpan) {
    return {
        kind: OpKind.Listener,
        target,
        targetSlot,
        tag,
        hostListener,
        name,
        handlerOps: new OpList(),
        handlerFnName: null,
        consumesDollarEvent: false,
        isAnimationListener: animationPhase !== null,
        animationPhase: animationPhase,
        sourceSpan,
        ...NEW_OP,
    };
}
export function createPipeOp(xref, slot, name) {
    return {
        kind: OpKind.Pipe,
        xref,
        handle: slot,
        name,
        ...NEW_OP,
        ...TRAIT_CONSUMES_SLOT,
    };
}
export function createNamespaceOp(namespace) {
    return {
        kind: OpKind.Namespace,
        active: namespace,
        ...NEW_OP,
    };
}
export function createProjectionDefOp(def) {
    return {
        kind: OpKind.ProjectionDef,
        def,
        ...NEW_OP,
    };
}
export function createProjectionOp(xref, selector, sourceSpan) {
    return {
        kind: OpKind.Projection,
        xref,
        handle: new SlotHandle(),
        selector,
        projectionSlotIndex: 0,
        attributes: [],
        localRefs: [],
        sourceSpan,
        ...NEW_OP,
        ...TRAIT_CONSUMES_SLOT,
    };
}
/**
 * Create an `ExtractedAttributeOp`.
 */
export function createExtractedAttributeOp(target, bindingKind, name, expression) {
    return {
        kind: OpKind.ExtractedAttribute,
        target,
        bindingKind,
        name,
        expression,
        ...NEW_OP,
    };
}
export function createDeferOp(xref, main, mainSlot, metadata, sourceSpan) {
    return {
        kind: OpKind.Defer,
        xref,
        handle: new SlotHandle(),
        mainView: main,
        mainSlot,
        loadingView: null,
        loadingSlot: null,
        loadingConfig: null,
        loadingMinimumTime: null,
        loadingAfterTime: null,
        placeholderView: null,
        placeholderSlot: null,
        placeholderConfig: null,
        placeholderMinimumTime: null,
        errorView: null,
        errorSlot: null,
        metadata,
        resolverFn: null,
        sourceSpan,
        ...NEW_OP,
        ...TRAIT_CONSUMES_SLOT,
        numSlotsUsed: 2,
    };
}
export function createDeferOnOp(defer, trigger, prefetch, sourceSpan) {
    return {
        kind: OpKind.DeferOn,
        defer,
        trigger,
        prefetch,
        sourceSpan,
        ...NEW_OP,
    };
}
/**
 * Create an `ExtractedMessageOp`.
 */
export function createI18nMessageOp(xref, i18nBlock, message, messagePlaceholder, params, postprocessingParams, needsPostprocessing) {
    return {
        kind: OpKind.I18nMessage,
        xref,
        i18nBlock,
        message,
        messagePlaceholder,
        params,
        postprocessingParams,
        needsPostprocessing,
        subMessages: [],
        ...NEW_OP,
    };
}
/**
 * Create an `I18nStartOp`.
 */
export function createI18nStartOp(xref, message, root) {
    return {
        kind: OpKind.I18nStart,
        xref,
        handle: new SlotHandle(),
        root: root ?? xref,
        message,
        messageIndex: null,
        subTemplateIndex: null,
        context: null,
        ...NEW_OP,
        ...TRAIT_CONSUMES_SLOT,
    };
}
/**
 * Create an `I18nEndOp`.
 */
export function createI18nEndOp(xref) {
    return {
        kind: OpKind.I18nEnd,
        xref,
        ...NEW_OP,
    };
}
/**
 * Creates an ICU start op.
 */
export function createIcuStartOp(xref, message, messagePlaceholder, sourceSpan) {
    return {
        kind: OpKind.IcuStart,
        xref,
        message,
        messagePlaceholder,
        context: null,
        sourceSpan,
        ...NEW_OP,
    };
}
/**
 * Creates an ICU end op.
 */
export function createIcuEndOp(xref) {
    return {
        kind: OpKind.IcuEnd,
        xref,
        ...NEW_OP,
    };
}
export function createI18nContextOp(contextKind, xref, i18nBlock, message, sourceSpan) {
    return {
        kind: OpKind.I18nContext,
        contextKind,
        xref,
        i18nBlock,
        message,
        sourceSpan,
        params: new Map(),
        postprocessingParams: new Map(),
        ...NEW_OP,
    };
}
export function literalOrArrayLiteral(value) {
    if (Array.isArray(value)) {
        return o.literalArr(value.map(literalOrArrayLiteral));
    }
    return o.literal(value, o.INFERRED_TYPE);
}
//# sourceMappingURL=data:application/json;base64,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