/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import * as o from '../output/output_ast';
import { Identifiers as R3 } from './r3_identifiers';
import { devOnlyGuardedExpression } from './util';
export function compileClassMetadata(metadata) {
    // Generate an ngDevMode guarded call to setClassMetadata with the class identifier and its
    // metadata.
    const fnCall = o.importExpr(R3.setClassMetadata).callFn([
        metadata.type,
        metadata.decorators,
        metadata.ctorParameters ?? o.literal(null),
        metadata.propDecorators ?? o.literal(null),
    ]);
    const iife = o.arrowFn([], [devOnlyGuardedExpression(fnCall).toStmt()]);
    return iife.callFn([]);
}
/**
 * Wraps the `setClassMetadata` function with extra logic that dynamically
 * loads dependencies from `@defer` blocks.
 *
 * Generates a call like this:
 * ```
 * setClassMetadataAsync(type, () => [
 *   import('./cmp-a').then(m => m.CmpA);
 *   import('./cmp-b').then(m => m.CmpB);
 * ], (CmpA, CmpB) => {
 *   setClassMetadata(type, decorators, ctorParameters, propParameters);
 * });
 * ```
 *
 * Similar to the `setClassMetadata` call, it's wrapped into the `ngDevMode`
 * check to tree-shake away this code in production mode.
 */
export function compileComponentClassMetadata(metadata, deferrableTypes) {
    if (deferrableTypes.size === 0) {
        // If there are no deferrable symbols - just generate a regular `setClassMetadata` call.
        return compileClassMetadata(metadata);
    }
    const dynamicImports = [];
    const importedSymbols = [];
    for (const [symbolName, importPath] of deferrableTypes) {
        // e.g. `(m) => m.CmpA`
        const innerFn = o.arrowFn([new o.FnParam('m', o.DYNAMIC_TYPE)], o.variable('m').prop(symbolName));
        // e.g. `import('./cmp-a').then(...)`
        const importExpr = (new o.DynamicImportExpr(importPath)).prop('then').callFn([innerFn]);
        dynamicImports.push(importExpr);
        importedSymbols.push(new o.FnParam(symbolName, o.DYNAMIC_TYPE));
    }
    // e.g. `() => [ ... ];`
    const dependencyLoadingFn = o.arrowFn([], o.literalArr(dynamicImports));
    // e.g. `setClassMetadata(...)`
    const setClassMetadataCall = o.importExpr(R3.setClassMetadata).callFn([
        metadata.type,
        metadata.decorators,
        metadata.ctorParameters ?? o.literal(null),
        metadata.propDecorators ?? o.literal(null),
    ]);
    // e.g. `(CmpA) => setClassMetadata(...)`
    const setClassMetaWrapper = o.arrowFn(importedSymbols, [setClassMetadataCall.toStmt()]);
    // Final `setClassMetadataAsync()` call with all arguments
    const setClassMetaAsync = o.importExpr(R3.setClassMetadataAsync).callFn([
        metadata.type, dependencyLoadingFn, setClassMetaWrapper
    ]);
    // Generate an ngDevMode guarded call to `setClassMetadataAsync` with
    // the class identifier and its metadata, so that this call can be tree-shaken.
    const iife = o.arrowFn([], [devOnlyGuardedExpression(setClassMetaAsync).toStmt()]);
    return iife.callFn([]);
}
//# sourceMappingURL=data:application/json;base64,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