/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { APP_BOOTSTRAP_LISTENER, ApplicationRef, inject, InjectionToken, makeStateKey, TransferState, ɵformatRuntimeError as formatRuntimeError, ɵperformanceMarkFeature as performanceMarkFeature, ɵtruncateMiddle as truncateMiddle, ɵwhenStable as whenStable } from '@angular/core';
import { of } from 'rxjs';
import { tap } from 'rxjs/operators';
import { HttpHeaders } from './headers';
import { HTTP_ROOT_INTERCEPTOR_FNS } from './interceptor';
import { HttpResponse } from './response';
/**
 * Keys within cached response data structure.
 */
export const BODY = 'b';
export const HEADERS = 'h';
export const STATUS = 's';
export const STATUS_TEXT = 'st';
export const URL = 'u';
export const RESPONSE_TYPE = 'rt';
const CACHE_OPTIONS = new InjectionToken(ngDevMode ? 'HTTP_TRANSFER_STATE_CACHE_OPTIONS' : '');
/**
 * A list of allowed HTTP methods to cache.
 */
const ALLOWED_METHODS = ['GET', 'HEAD'];
export function transferCacheInterceptorFn(req, next) {
    const { isCacheActive, ...globalOptions } = inject(CACHE_OPTIONS);
    const { transferCache: requestOptions, method: requestMethod } = req;
    // In the following situations we do not want to cache the request
    if (!isCacheActive ||
        // POST requests are allowed either globally or at request level
        (requestMethod === 'POST' && !globalOptions.includePostRequests && !requestOptions) ||
        (requestMethod !== 'POST' && !ALLOWED_METHODS.includes(requestMethod)) ||
        requestOptions === false || //
        (globalOptions.filter?.(req)) === false) {
        return next(req);
    }
    const transferState = inject(TransferState);
    const storeKey = makeCacheKey(req);
    const response = transferState.get(storeKey, null);
    let headersToInclude = globalOptions.includeHeaders;
    if (typeof requestOptions === 'object' && requestOptions.includeHeaders) {
        // Request-specific config takes precedence over the global config.
        headersToInclude = requestOptions.includeHeaders;
    }
    if (response) {
        const { [BODY]: undecodedBody, [RESPONSE_TYPE]: responseType, [HEADERS]: httpHeaders, [STATUS]: status, [STATUS_TEXT]: statusText, [URL]: url } = response;
        // Request found in cache. Respond using it.
        let body = undecodedBody;
        switch (responseType) {
            case 'arraybuffer':
                body = new TextEncoder().encode(undecodedBody).buffer;
                break;
            case 'blob':
                body = new Blob([undecodedBody]);
                break;
        }
        // We want to warn users accessing a header provided from the cache
        // That HttpTransferCache alters the headers
        // The warning will be logged a single time by HttpHeaders instance
        let headers = new HttpHeaders(httpHeaders);
        if (typeof ngDevMode === 'undefined' || ngDevMode) {
            // Append extra logic in dev mode to produce a warning when a header
            // that was not transferred to the client is accessed in the code via `get`
            // and `has` calls.
            headers = appendMissingHeadersDetection(req.url, headers, headersToInclude ?? []);
        }
        return of(new HttpResponse({
            body,
            headers,
            status,
            statusText,
            url,
        }));
    }
    // Request not found in cache. Make the request and cache it.
    return next(req).pipe(tap((event) => {
        if (event instanceof HttpResponse) {
            transferState.set(storeKey, {
                [BODY]: event.body,
                [HEADERS]: getFilteredHeaders(event.headers, headersToInclude),
                [STATUS]: event.status,
                [STATUS_TEXT]: event.statusText,
                [URL]: event.url || '',
                [RESPONSE_TYPE]: req.responseType,
            });
        }
    }));
}
function getFilteredHeaders(headers, includeHeaders) {
    if (!includeHeaders) {
        return {};
    }
    const headersMap = {};
    for (const key of includeHeaders) {
        const values = headers.getAll(key);
        if (values !== null) {
            headersMap[key] = values;
        }
    }
    return headersMap;
}
function makeCacheKey(request) {
    // make the params encoded same as a url so it's easy to identify
    const { params, method, responseType, url } = request;
    const encodedParams = params.keys().sort().map((k) => `${k}=${params.getAll(k)}`).join('&');
    const key = method + '.' + responseType + '.' + url + '?' + encodedParams;
    const hash = generateHash(key);
    return makeStateKey(hash);
}
/**
 * A method that returns a hash representation of a string using a variant of DJB2 hash
 * algorithm.
 *
 * This is the same hashing logic that is used to generate component ids.
 */
function generateHash(value) {
    let hash = 0;
    for (const char of value) {
        hash = Math.imul(31, hash) + char.charCodeAt(0) << 0;
    }
    // Force positive number hash.
    // 2147483647 = equivalent of Integer.MAX_VALUE.
    hash += 2147483647 + 1;
    return hash.toString();
}
/**
 * Returns the DI providers needed to enable HTTP transfer cache.
 *
 * By default, when using server rendering, requests are performed twice: once on the server and
 * other one on the browser.
 *
 * When these providers are added, requests performed on the server are cached and reused during the
 * bootstrapping of the application in the browser thus avoiding duplicate requests and reducing
 * load time.
 *
 */
export function withHttpTransferCache(cacheOptions) {
    return [
        {
            provide: CACHE_OPTIONS,
            useFactory: () => {
                performanceMarkFeature('NgHttpTransferCache');
                return { isCacheActive: true, ...cacheOptions };
            }
        },
        {
            provide: HTTP_ROOT_INTERCEPTOR_FNS,
            useValue: transferCacheInterceptorFn,
            multi: true,
            deps: [TransferState, CACHE_OPTIONS]
        },
        {
            provide: APP_BOOTSTRAP_LISTENER,
            multi: true,
            useFactory: () => {
                const appRef = inject(ApplicationRef);
                const cacheState = inject(CACHE_OPTIONS);
                return () => {
                    whenStable(appRef).then(() => {
                        cacheState.isCacheActive = false;
                    });
                };
            }
        }
    ];
}
/**
 * This function will add a proxy to an HttpHeader to intercept calls to get/has
 * and log a warning if the header entry requested has been removed
 */
function appendMissingHeadersDetection(url, headers, headersToInclude) {
    const warningProduced = new Set();
    return new Proxy(headers, {
        get(target, prop) {
            const value = Reflect.get(target, prop);
            const methods = new Set(['get', 'has', 'getAll']);
            if (typeof value !== 'function' || !methods.has(prop)) {
                return value;
            }
            return (headerName) => {
                // We log when the key has been removed and a warning hasn't been produced for the header
                const key = (prop + ':' + headerName).toLowerCase(); // e.g. `get:cache-control`
                if (!headersToInclude.includes(headerName) && !warningProduced.has(key)) {
                    warningProduced.add(key);
                    const truncatedUrl = truncateMiddle(url);
                    // TODO: create Error guide for this warning
                    console.warn(formatRuntimeError(2802 /* RuntimeErrorCode.HEADERS_ALTERED_BY_TRANSFER_CACHE */, `Angular detected that the \`${headerName}\` header is accessed, but the value of the header ` +
                        `was not transferred from the server to the client by the HttpTransferCache. ` +
                        `To include the value of the \`${headerName}\` header for the \`${truncatedUrl}\` request, ` +
                        `use the \`includeHeaders\` list. The \`includeHeaders\` can be defined either ` +
                        `on a request level by adding the \`transferCache\` parameter, or on an application ` +
                        `level by adding the \`httpCacheTransfer.includeHeaders\` argument to the ` +
                        `\`provideClientHydration()\` call. `));
                }
                // invoking the original method
                return value.apply(target, [headerName]);
            };
        }
    });
}
//# sourceMappingURL=data:application/json;base64,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