/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ɵRuntimeError as RuntimeError } from '@angular/core';
const LINE_START = '\n - ';
export function invalidTimingValue(exp) {
    return new RuntimeError(3000 /* RuntimeErrorCode.INVALID_TIMING_VALUE */, ngDevMode && `The provided timing value "${exp}" is invalid.`);
}
export function negativeStepValue() {
    return new RuntimeError(3100 /* RuntimeErrorCode.NEGATIVE_STEP_VALUE */, ngDevMode && 'Duration values below 0 are not allowed for this animation step.');
}
export function negativeDelayValue() {
    return new RuntimeError(3101 /* RuntimeErrorCode.NEGATIVE_DELAY_VALUE */, ngDevMode && 'Delay values below 0 are not allowed for this animation step.');
}
export function invalidStyleParams(varName) {
    return new RuntimeError(3001 /* RuntimeErrorCode.INVALID_STYLE_PARAMS */, ngDevMode &&
        `Unable to resolve the local animation param ${varName} in the given list of values`);
}
export function invalidParamValue(varName) {
    return new RuntimeError(3003 /* RuntimeErrorCode.INVALID_PARAM_VALUE */, ngDevMode && `Please provide a value for the animation param ${varName}`);
}
export function invalidNodeType(nodeType) {
    return new RuntimeError(3004 /* RuntimeErrorCode.INVALID_NODE_TYPE */, ngDevMode && `Unable to resolve animation metadata node #${nodeType}`);
}
export function invalidCssUnitValue(userProvidedProperty, value) {
    return new RuntimeError(3005 /* RuntimeErrorCode.INVALID_CSS_UNIT_VALUE */, ngDevMode && `Please provide a CSS unit value for ${userProvidedProperty}:${value}`);
}
export function invalidTrigger() {
    return new RuntimeError(3006 /* RuntimeErrorCode.INVALID_TRIGGER */, ngDevMode &&
        'animation triggers cannot be prefixed with an `@` sign (e.g. trigger(\'@foo\', [...]))');
}
export function invalidDefinition() {
    return new RuntimeError(3007 /* RuntimeErrorCode.INVALID_DEFINITION */, ngDevMode && 'only state() and transition() definitions can sit inside of a trigger()');
}
export function invalidState(metadataName, missingSubs) {
    return new RuntimeError(3008 /* RuntimeErrorCode.INVALID_STATE */, ngDevMode &&
        `state("${metadataName}", ...) must define default values for all the following style substitutions: ${missingSubs.join(', ')}`);
}
export function invalidStyleValue(value) {
    return new RuntimeError(3002 /* RuntimeErrorCode.INVALID_STYLE_VALUE */, ngDevMode && `The provided style string value ${value} is not allowed.`);
}
export function invalidProperty(prop) {
    return new RuntimeError(3009 /* RuntimeErrorCode.INVALID_PROPERTY */, ngDevMode &&
        `The provided animation property "${prop}" is not a supported CSS property for animations`);
}
export function invalidParallelAnimation(prop, firstStart, firstEnd, secondStart, secondEnd) {
    return new RuntimeError(3010 /* RuntimeErrorCode.INVALID_PARALLEL_ANIMATION */, ngDevMode &&
        `The CSS property "${prop}" that exists between the times of "${firstStart}ms" and "${firstEnd}ms" is also being animated in a parallel animation between the times of "${secondStart}ms" and "${secondEnd}ms"`);
}
export function invalidKeyframes() {
    return new RuntimeError(3011 /* RuntimeErrorCode.INVALID_KEYFRAMES */, ngDevMode && `keyframes() must be placed inside of a call to animate()`);
}
export function invalidOffset() {
    return new RuntimeError(3012 /* RuntimeErrorCode.INVALID_OFFSET */, ngDevMode && `Please ensure that all keyframe offsets are between 0 and 1`);
}
export function keyframeOffsetsOutOfOrder() {
    return new RuntimeError(3200 /* RuntimeErrorCode.KEYFRAME_OFFSETS_OUT_OF_ORDER */, ngDevMode && `Please ensure that all keyframe offsets are in order`);
}
export function keyframesMissingOffsets() {
    return new RuntimeError(3202 /* RuntimeErrorCode.KEYFRAMES_MISSING_OFFSETS */, ngDevMode && `Not all style() steps within the declared keyframes() contain offsets`);
}
export function invalidStagger() {
    return new RuntimeError(3013 /* RuntimeErrorCode.INVALID_STAGGER */, ngDevMode && `stagger() can only be used inside of query()`);
}
export function invalidQuery(selector) {
    return new RuntimeError(3014 /* RuntimeErrorCode.INVALID_QUERY */, ngDevMode &&
        `\`query("${selector}")\` returned zero elements. (Use \`query("${selector}", { optional: true })\` if you wish to allow this.)`);
}
export function invalidExpression(expr) {
    return new RuntimeError(3015 /* RuntimeErrorCode.INVALID_EXPRESSION */, ngDevMode && `The provided transition expression "${expr}" is not supported`);
}
export function invalidTransitionAlias(alias) {
    return new RuntimeError(3016 /* RuntimeErrorCode.INVALID_TRANSITION_ALIAS */, ngDevMode && `The transition alias value "${alias}" is not supported`);
}
export function validationFailed(errors) {
    return new RuntimeError(3500 /* RuntimeErrorCode.VALIDATION_FAILED */, ngDevMode && `animation validation failed:\n${errors.map(err => err.message).join('\n')}`);
}
export function buildingFailed(errors) {
    return new RuntimeError(3501 /* RuntimeErrorCode.BUILDING_FAILED */, ngDevMode && `animation building failed:\n${errors.map(err => err.message).join('\n')}`);
}
export function triggerBuildFailed(name, errors) {
    return new RuntimeError(3404 /* RuntimeErrorCode.TRIGGER_BUILD_FAILED */, ngDevMode &&
        `The animation trigger "${name}" has failed to build due to the following errors:\n - ${errors.map(err => err.message).join('\n - ')}`);
}
export function animationFailed(errors) {
    return new RuntimeError(3502 /* RuntimeErrorCode.ANIMATION_FAILED */, ngDevMode &&
        `Unable to animate due to the following errors:${LINE_START}${errors.map(err => err.message).join(LINE_START)}`);
}
export function registerFailed(errors) {
    return new RuntimeError(3503 /* RuntimeErrorCode.REGISTRATION_FAILED */, ngDevMode &&
        `Unable to build the animation due to the following errors: ${errors.map(err => err.message).join('\n')}`);
}
export function missingOrDestroyedAnimation() {
    return new RuntimeError(3300 /* RuntimeErrorCode.MISSING_OR_DESTROYED_ANIMATION */, ngDevMode && 'The requested animation doesn\'t exist or has already been destroyed');
}
export function createAnimationFailed(errors) {
    return new RuntimeError(3504 /* RuntimeErrorCode.CREATE_ANIMATION_FAILED */, ngDevMode &&
        `Unable to create the animation due to the following errors:${errors.map(err => err.message).join('\n')}`);
}
export function missingPlayer(id) {
    return new RuntimeError(3301 /* RuntimeErrorCode.MISSING_PLAYER */, ngDevMode && `Unable to find the timeline player referenced by ${id}`);
}
export function missingTrigger(phase, name) {
    return new RuntimeError(3302 /* RuntimeErrorCode.MISSING_TRIGGER */, ngDevMode &&
        `Unable to listen on the animation trigger event "${phase}" because the animation trigger "${name}" doesn\'t exist!`);
}
export function missingEvent(name) {
    return new RuntimeError(3303 /* RuntimeErrorCode.MISSING_EVENT */, ngDevMode &&
        `Unable to listen on the animation trigger "${name}" because the provided event is undefined!`);
}
export function unsupportedTriggerEvent(phase, name) {
    return new RuntimeError(3400 /* RuntimeErrorCode.UNSUPPORTED_TRIGGER_EVENT */, ngDevMode &&
        `The provided animation trigger event "${phase}" for the animation trigger "${name}" is not supported!`);
}
export function unregisteredTrigger(name) {
    return new RuntimeError(3401 /* RuntimeErrorCode.UNREGISTERED_TRIGGER */, ngDevMode && `The provided animation trigger "${name}" has not been registered!`);
}
export function triggerTransitionsFailed(errors) {
    return new RuntimeError(3402 /* RuntimeErrorCode.TRIGGER_TRANSITIONS_FAILED */, ngDevMode &&
        `Unable to process animations due to the following failed trigger transitions\n ${errors.map(err => err.message).join('\n')}`);
}
export function triggerParsingFailed(name, errors) {
    return new RuntimeError(3403 /* RuntimeErrorCode.TRIGGER_PARSING_FAILED */, ngDevMode &&
        `Animation parsing for the ${name} trigger have failed:${LINE_START}${errors.map(err => err.message).join(LINE_START)}`);
}
export function transitionFailed(name, errors) {
    return new RuntimeError(3505 /* RuntimeErrorCode.TRANSITION_FAILED */, ngDevMode && `@${name} has failed due to:\n ${errors.map(err => err.message).join('\n- ')}`);
}
//# sourceMappingURL=data:application/json;base64,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